-- ScoreRulesTests.lua
-- Minimal unit tests for ScoreRules (no UI, pure functions)

T = {}

-- --- tiny assertion helpers -----------------------------------------------

local function fail(msg) error(msg, 2) end

local function expectEq(actual, expected, label)
    if actual ~= expected then
        fail(string.format("FAILED: %s (expected %s, got %s)", label, tostring(expected), tostring(actual)))
    end
end

local function expectTrue(cond, label)
    if not cond then fail("FAILED: " .. label .. " (expected true)") end
end

local function expectFalse(cond, label)
    if cond then fail("FAILED: " .. label .. " (expected false)") end
end

-- --- fixtures --------------------------------------------------------------

local function newTeam()
    return {
        players = { {bid=0,took=0}, {bid=0,took=0} }, -- spades irrelevant here
        hearts = 0, queensSpades = false, moonShot = false,
        _oppMoonBonus = 0
    }
end

local function newTeams()
    return { newTeam(), newTeam() }
end

local function heartsScore(team)
    local h = ScoreRules.heartsHand(team)
    return h.handScore, h
end

-- --- tests ----------------------------------------------------------------
-- 1) setting moon sets other teams hearts to 0
function T.test_moon_sets_other_hearts_to_0()
    local teams = newTeams()
    teams[2].hearts = 7 -- prefill to prove it gets cleared
    teams[1].moonShot = true
    ScoreRules.syncHeartsMoon(teams)
    expectEq(teams[2].hearts, 0, "other team hearts forced to 0")
end

-- 2) setting moon clears queen from other team
function T.test_moon_clears_other_queen()
    local teams = newTeams()
    teams[2].queensSpades = true
    teams[1].moonShot = true
    ScoreRules.syncHeartsMoon(teams)
    expectFalse(teams[2].queensSpades, "other team queen cleared")
end

-- 3) setting moon sets team’s hearts to 13 and queen to true
function T.test_moon_sets_team_hearts_13_and_queen_true()
    local teams = newTeams()
    teams[1].moonShot = true
    ScoreRules.syncHeartsMoon(teams)
    expectEq(teams[1].hearts, 13, "moon team hearts forced to 13")
    expectTrue(teams[1].queensSpades, "moon team queen forced true")
end

-- 4) setting moon sets team’s hearts score to 0
function T.test_moon_sets_team_hearts_score_0()
    local teams = newTeams()
    teams[1].moonShot = true
    ScoreRules.syncHeartsMoon(teams)
    local s1 = heartsScore(teams[1])
    expectEq(s1, 0, "moon team hearts score == 0")
end

-- 5) setting moon sets other team’s hearts score to 104
function T.test_moon_sets_other_hearts_score_104()
    local teams = newTeams()
    teams[1].moonShot = true
    ScoreRules.syncHeartsMoon(teams)
    local s2 = heartsScore(teams[2])
    expectEq(s2, 104, "other team gets +104 when opponent moons")
end

-- 6) taking queen plus 13 cards but no moon sets hearts score to 104
function T.test_queen_plus_13_no_moon_scores_104()
    local teams = newTeams()
    teams[1].hearts = 13
    teams[1].queensSpades = true
    teams[1].moonShot = false
    ScoreRules.syncHeartsMoon(teams)
    local s1 = heartsScore(teams[1])
    expectEq(s1, 104, "13 hearts + queen, no moon => 104")
end

-- --- runner ----------------------------------------------------------------

function T.run()
    local list = {
        {"moon_sets_other_hearts_to_0",                 T.test_moon_sets_other_hearts_to_0},
        {"moon_clears_other_queen",                     T.test_moon_clears_other_queen},
        {"moon_sets_team_hearts_13_and_queen_true",     T.test_moon_sets_team_hearts_13_and_queen_true},
        {"moon_sets_team_hearts_score_0",               T.test_moon_sets_team_hearts_score_0},
        {"moon_sets_other_hearts_score_104",            T.test_moon_sets_other_hearts_score_104},
        {"queen_plus_13_no_moon_scores_104",            T.test_queen_plus_13_no_moon_scores_104},
    }
    local ok, failed = 0, 0
    for _,pair in ipairs(list) do
        local name, fn = pair[1], pair[2]
        local status, err = pcall(fn)
        if status then
            ok = ok + 1
            print("✔︎ PASS:", name)
        else
            failed = failed + 1
            print("✘ FAIL:", name, "\n   ", err)
        end
    end
    print(string.format("\nRESULT: %d passed, %d failed", ok, failed))
end