

-- IncrementingCell.lua
-- Tap to increment / long-press removed elsewhere.
-- Now cycles: 0..13 → UNSET (“--”) → 0 (and reverse when decrementing)

IncrementingCell = class()

function IncrementingCell:init(x, y, w, h, initialValue)
    self.x, self.y, self.w, self.h = x, y, w, h
    self.min, self.max, self.wrap = 0, 13, true
    
    -- value/hasSet semantics:
    --   hasSet=false => display "--" and treat as nil in syncBack()
    self.hasSet = (initialValue ~= nil)
    self.value  = self.hasSet and initialValue or 0
    
    self.fontSize = 20
    self.isPressed = false
    
    -- theme (can be overridden from outside)
    self.colStroke    = self.colStroke    or Theme.gridLine
    self.colBg        = self.colBg        or Theme.cellBg
    self.colBgPressed = self.colBgPressed or Theme.cellBgPressed
    self.colText      = self.colText      or Theme.textAccentBlue
    self.colTextUnset = self.colTextUnset or Theme.textDisabled
    
    -- sensors
    self.sensorPress = Sensor{ parent = self }
    self.sensorTap   = Sensor{ parent = self }
    
    self.sensorPress:onTouch(function(ev)
        self.isPressed = (ev.state and true) or false
    end)
    
    self.sensorTap:onTap(function()
        self:step(1)   -- single tap steps up
    end)
end

function IncrementingCell:set(v)
    v = math.floor(tonumber(v) or 0)
    v = math.max(self.min, math.min(self.max, v))
    self.value, self.hasSet = v, true
end

function IncrementingCell:unset()
    self.hasSet = false
end

-- New: stepping includes the UNSET state in the ring
-- order when going up: 0,1,...,13, UNSET, 0,1,...
-- order when going down: ...,1,0, UNSET, 13,12,...
function IncrementingCell:step(dir)
    dir = (dir and dir < 0) and -1 or 1
    if self.hasSet then
        local v = self.value + dir
        if v > self.max then
            -- wrap to UNSET when going past 13
            self.hasSet = false
        elseif v < self.min then
            -- wrap to UNSET when going below 0
            self.hasSet = false
        else
            self.value = v
        end
    else
        -- currently UNSET → jump to end based on direction
        if dir > 0 then
            self.value, self.hasSet = self.min, true   -- UNSET -> 0
        else
            self.value, self.hasSet = self.max, true   -- UNSET -> 13
        end
    end
end

function IncrementingCell:decrement() self:step(0-1) end
function IncrementingCell:increment() self:step(1) end

function IncrementingCell:inbox(t)
    return (t.x >= self.x and t.x <= self.x + self.w
    and t.y >= self.y and t.y <= self.y + self.h)
end

function IncrementingCell:touched(t)
    self.sensorPress:touched(t)
    self.sensorTap:touched(t)
end

function IncrementingCell:draw()
    pushStyle()
    rectMode(CORNER)
    if self.isPressed then
        fill(self.colBgPressed) ; stroke(self.colStroke) ; strokeWidth(2)
    else
        fill(self.colBg) ; stroke(self.colStroke) ; strokeWidth(1)
    end
    rect(self.x, self.y, self.w, self.h)
    
    -- text
    local label = self.hasSet and tostring(self.value) or "--"
    fill(self.hasSet and self.colText or self.colTextUnset)
    font("HelveticaNeue-Bold")
    fontSize(self.fontSize)
    textAlign(CENTER)
    text(label, self.x + self.w/2, self.y + self.h/2)
    popStyle()
end