-- ScoreSheets.lua
ScoreSheets = class()

function ScoreSheets:init(makeTeams)
    self._scrollTween = nil
    self.makeTeams = makeTeams or self._defaultTeams
    self.tables    = {}
    self.scrollY   = 0
    
    -- Start with ONE table
    table.insert(self.tables, ScoreTable(self.makeTeams(1)))
    
    -- Scroll sensor (whole screen)
    self.scroll = {
        sensor   = Sensor{ parent = {x = WIDTH/2, y = HEIGHT/2, w = WIDTH, h = HEIGHT}, xywhMode = CENTER },
        mode     = "idle",
        startY   = 0,
        startSY  = 0
    }
    local DRAG_THRESH = 10
    self.scroll.sensor:onTouched(function(ev)
        local t = ev.touch
        if t.state == BEGAN then
            -- cancel any in-flight scroll tween so the user takes control
            if self._scrollTween then tween.stop(self._scrollTween); self._scrollTween = nil end
            self.scroll.mode   = "maybe-drag"
            self.scroll.startY = t.y
            self.scroll.startSY= self.scrollY
            self.scroll.sensor.doNotInterceptTouches = true
        elseif t.state == MOVING and self.scroll.mode ~= "idle" then
            local dy = t.y - self.scroll.startY
            if self.scroll.mode == "maybe-drag" and math.abs(dy) > DRAG_THRESH then
                self.scroll.mode = "dragging"
                self.scroll.sensor.doNotInterceptTouches = false
            end
            if self.scroll.mode == "dragging" then
                self.scrollY = self.scroll.startSY + dy
                self:_clampScroll()
            end
        elseif t.state == ENDED or t.state == CANCELLED then
            self.scroll.mode = "idle"
            self.scroll.sensor.doNotInterceptTouches = true
        end
    end)
    
    -- New Hand button (fixed UI)
    self.newBtn = {
        x = WIDTH - 140, y = 22, w = 120, h = 44,
        sensor = Sensor{ parent = {x = WIDTH - 80, y = 44, w = 120, h = 44}, xywhMode = CENTER }
    }
    self.newBtn.sensor:onTap(function()
        self:_addHand()
    end)
    
    -- Reset All button (fixed UI)
    self.resetBtn = {
        w = 120, h = 44,
        sensor = Sensor{ parent = {x=0, y=0, w=120, h=44}, xywhMode = CORNER }
    }
    self.resetBtn.sensor:onTap(function()
        self:_resetAll()
    end)
    
end

-- Default teams for the very first table/hand
function ScoreSheets:_defaultTeams()
    local function team()
        return {
            players = { {name="A", bid=nil, took=nil}, {name="B", bid=nil, took=nil} },
            hearts=nil, queensSpades=false, moonShot=false,
            -- cumulative/running
            spadesTotal=0, heartsTotal=0, gameTotal=0, allBags=0,
            -- per-hand derived (optional — convenient to keep here)
            spadesScore=nil, heartsScore=nil, handBags=nil, _oppMoonBonus=0
        }
    end
    return { team(), team() }
end

-- Deep-ish copy just the cumulative state; reset per-hand inputs
local function newTeamsFromPrevious(prevTeams)
    local function cloneTotals(dst, src)
        dst.spadesTotal = src.spadesTotal or 0
        dst.heartsTotal = src.heartsTotal or 0
        dst.gameTotal   = src.gameTotal   or 0
        dst.allBags     = src.allBags     or 0
    end
    local function newTeamFrom(prev)
        local t = {
            players = {
                { name = prev.players[1].name, bid=nil, took=nil },
                { name = prev.players[2].name, bid=nil, took=nil },
            },
            hearts=nil, queensSpades=false, moonShot=false,
            spadesScore=nil, heartsScore=nil, handBags=nil,
            _oppMoonBonus=0
        }
        cloneTotals(t, prev)
        return t
    end
    return { newTeamFrom(prevTeams[1]), newTeamFrom(prevTeams[2]) }
end

function ScoreSheets:_addHand()
    local last = self.tables[#self.tables]
    local teams = newTeamsFromPrevious(last.teams)
    table.insert(self.tables, ScoreTable(teams))
    
    local stepH, gapH = self:_stackMetrics()
    local d = stepH + gapH
    local targetY = (#self.tables - 1) * d  -- center the NEW hand
    self:_scrollTo(targetY, 0.4)            -- smooth scroll ~0.4s
end

-- Reset everything to a single fresh hand, preserving player names
function ScoreSheets:_resetAll()
    local first = self.tables[1] and self.tables[1].teams
    local function freshFrom(team)
        return {
            players = {
                { name = team and team.players[1].name or "A", bid=nil, took=nil },
                { name = team and team.players[2].name or "B", bid=nil, took=nil },
            },
            hearts=nil, queensSpades=false, moonShot=false,
            -- running totals cleared
            spadesTotal=0, heartsTotal=0, gameTotal=0, allBags=0,
            -- per-hand derived cleared
            spadesScore=nil, heartsScore=nil, handBags=nil, _oppMoonBonus=0
        }
    end
    
    local teams = {
        freshFrom(first and first[1]),
        freshFrom(first and first[2]),
    }
    
    self.tables = { ScoreTable(teams) }
    self.scrollY = 0
end

-- how far the stack extends below the first (in pixels)
function ScoreSheets:_contentSpan()
    local stepH, gapH = self:_stackMetrics()
    local d = stepH + gapH
    return math.max(0, (#self.tables - 1) * d)
end

function ScoreSheets:_clampScroll()
    local span = self:_contentSpan()
    if self.scrollY < 0     then self.scrollY = 0 end
    if self.scrollY > span  then self.scrollY = span end
end

function ScoreSheets:_scrollTo(targetY, duration)
    local span = self:_contentSpan()
    local clamped = math.max(0, math.min(span, targetY))
    if self._scrollTween then tween.stop(self._scrollTween) end
    -- duration default ~0.35s; linear tween is fine (omit easing arg for safety)
    self._scrollTween = tween(duration or 0.35, self, { scrollY = clamped }, nil, function()
        self._scrollTween = nil
    end)
end

-- Accumulate running totals for table i using table (i-1)
local function accumulateFromPrevious(prevTeams, thisTeams)
    -- Hearts: normalize first so scores are computed from forced state
    ScoreRules.syncHeartsMoon(thisTeams)
    
    -- SPADES per-hand (nil if not ready)
    local s1 = ScoreRules.spadesHand(thisTeams[1])
    local s2 = ScoreRules.spadesHand(thisTeams[2])
    
    -- HEARTS per-hand (nil if not ready)
    local h1 = ScoreRules.heartsHand(thisTeams[1])
    local h2 = ScoreRules.heartsHand(thisTeams[2])
    
    -- Write back per-hand visible scores (so right table can read them)
    thisTeams[1].spadesScore = s1.handScore
    thisTeams[2].spadesScore = s2.handScore
    thisTeams[1].heartsScore = h1.handScore
    thisTeams[2].heartsScore = h2.handScore
    thisTeams[1].handBags    = s1.handBags
    thisTeams[2].handBags    = s2.handBags
    
    -- Carry totals from prev
    for ti=1,2 do
        local prev = prevTeams[ti]
        local cur  = thisTeams[ti]
        cur.spadesTotal = (prev.spadesTotal or 0) + (cur.spadesScore or 0)
        cur.heartsTotal = (prev.heartsTotal or 0) + (cur.heartsScore or 0)
        -- bags: add, apply sandbag penalty
        local allBags = (prev.allBags or 0) + (cur.handBags or 0)
        local rem, penalty = ScoreRules.applySandbagPenalty(allBags, 10, 100)
        cur.allBags = rem
        cur.spadesTotal = cur.spadesTotal - (penalty or 0)
        -- grand
        cur.gameTotal = cur.spadesTotal - cur.heartsTotal
    end
end

-- helper right above accumulateFromPrevious or near it
local function zeroPrevTeams()
    return {
        { spadesTotal = 0, heartsTotal = 0, allBags = 0 },
        { spadesTotal = 0, heartsTotal = 0, allBags = 0 }
    }
end

function ScoreSheets:draw()
    background(46)
    for i = 1, #self.tables do
        if i == 1 then
            -- First hand: accumulate from zeros so per-hand values appear in totals
            accumulateFromPrevious(zeroPrevTeams(), self.tables[i].teams)
        else
            accumulateFromPrevious(self.tables[i-1].teams, self.tables[i].teams)
        end
    end
    
    -- Draw all tables with vertical offset & scroll
    do
        local stepH, gapH = self:_stackMetrics()
        local d = stepH + gapH
        for i = 1, #self.tables do
            pushMatrix()
            -- first hand at scrollY==0 is centered; later hands are drawn LOWER (negative offset)
            translate(0, - (i-1) * d + self.scrollY)
            self.tables[i]:draw()
            popMatrix()
        end
    end
    
    -- “Reset All” button (upper-left above the first table, scrolls with it)
    do
        local t = self.tables[1]
        if t and t.metrics then
            local m = t.metrics
            local bw, bh = self.resetBtn.w, self.resetBtn.h
            
            -- Anchor to the first table’s left edge; sit higher above the header.
            local baseX = m.innerX
            local baseY = m.innerY + m.tablesH + (m.leftRowH * 0.75) + 8  -- higher than before
            
            -- Make it scroll along with the first sheet
            local bx = baseX
            local by = baseY + self.scrollY
            
            -- keep the sensor aligned in absolute (screen) space
            self.resetBtn.sensor:setParent{
                parent = { x = bx, y = by, w = bw, h = bh },
                xywhMode = CORNER
            }
            
            pushStyle()
            -- Different color (warm orange), slightly translucent
            fill(255, 140, 0, 235)
            rect(bx, by, bw, bh, 10)
            fill(255)
            fontSize(18)
            textAlign(CENTER)
            text("Reset All", bx + bw/2, by + bh/2)
            popStyle()
        end
    end
    
    -- Fixed “New Hand” button
    pushStyle()
    fill(40, 160, 255, 220)
    rect(self.newBtn.x, self.newBtn.y, self.newBtn.w, self.newBtn.h, 10)
    fill(255)
    fontSize(20)
    text("New Hand", self.newBtn.x + self.newBtn.w/2, self.newBtn.y + self.newBtn.h/2)
    popStyle()
end

function ScoreSheets:touched(t)
    -- 1) Scroll lens first
    self.scroll.sensor:touched(t)
    if self.scroll.mode == "dragging" then return true end
    
    -- 2) New Hand button (fixed)
    if self.newBtn.sensor:touched(t) then return true end
    -- 2b) Reset All button (fixed)
    if self.resetBtn.sensor:touched(t) then return true end
    
    -- 3) Forward touch to the visible table under the finger
    do
        local stepH, gapH = self:_stackMetrics()
        local d = stepH + gapH
        for i = 1, #self.tables do
            -- map the finger to this table's local space (inverse of translate in :draw)
            local tt = {
                id=t.id, state=t.state, tapCount=t.tapCount,
                x=t.x, y=t.y - ( - (i-1)*d + self.scrollY ),  -- un-translate
                deltaX=t.deltaX, deltaY=t.deltaY
            }
            -- Only pass touches to a table if the finger is roughly over its block.
            -- The block’s vertical span is stepH (the table height).
            local centerY = HEIGHT/2
            local topY    = centerY + stepH/2
            local botY    = centerY - stepH/2
            if tt.y >= botY and tt.y <= topY then
                if self.tables[i]:touched(tt) then return true end
            end
        end
    end
    return false
end
-- returns (stepH, gapH)
function ScoreSheets:_stackMetrics()
    -- Ensure we have fresh layout metrics from the first table
    local t = self.tables[1]
    if t and t.layout then t:layout() end
    
    -- The vertical extent of a table block:
    -- ScoreTable uses layout.overallHeightPercent of the screen height
    local stepH = HEIGHT * (layout.overallHeightPercent / 100)
    
    -- A compact gap ≈ one left-row cell height
    local gapH  = (t and t.metrics and t.metrics.leftRowH) and (t.metrics.leftRowH * 0.9) or 16
    
    return stepH, gapH
end