-- ScoreLedger.lua
-- Maintains a list of completed hands and a "live" hand (from UI).
-- Recomputes running totals and writes them back into teams[*].

ScoreLedger = class()

function ScoreLedger:init(rules, teams)
    self.rules = rules or ScoreRules
    self.teams = teams
    self.hands = {}   -- array of { team1 = {...}, team2 = {...} }
end

-- Optionally call when a hand is done. It snapshots the *current* values from teams.
function ScoreLedger:finalizeHand()
    table.insert(self.hands, self:_snapshotLive())
end

-- Internal: snapshot the current "live" hand from teams (without mutating anything)
function ScoreLedger:_snapshotLive()
    local t = self.teams
    local function copyTeam(team)
        return {
            players = {
                { bid = team.players[1].bid or 0, took = team.players[1].took or 0 },
                { bid = team.players[2].bid or 0, took = team.players[2].took or 0 },
            },
            hearts       = team.hearts or 0,
            queensSpades = team.queensSpades or false,
            moonShot     = team.moonShot or false,
        }
    end
    return { team1 = copyTeam(t[1]), team2 = copyTeam(t[2]) }
end

-- Recompute totals using: all completed hands + current live hand
function ScoreLedger:recompute()
    local R = self.rules
    local totals = {
        -- per team accumulators
        [1] = { spadesTotal=0, heartsTotal=0, allBags=0 },
        [2] = { spadesTotal=0, heartsTotal=0, allBags=0 },
    }
    
    local function addHandFrame(teamIdx, frameTeam)
        -- SPADES
        local s = R.spadesHand(frameTeam)
        totals[teamIdx].spadesTotal = totals[teamIdx].spadesTotal + s.handScore
        totals[teamIdx].allBags     = totals[teamIdx].allBags + s.over
        -- HEARTS
        local h = R.heartsHand(frameTeam)
        totals[teamIdx].heartsTotal = totals[teamIdx].heartsTotal + h.handScore
    end
    
    -- 1) accumulate completed hands
    for _,h in ipairs(self.hands) do
        addHandFrame(1, h.team1)
        addHandFrame(2, h.team2)
    end
    
    -- 2) Include the live hand (current UI state)
    local live = self:_snapshotLive()
    addHandFrame(1, live.team1)
    addHandFrame(2, live.team2)
    
    -- 3) apply sandbag penalties (Spades)
    for i=1,2 do
        local newAllBags, penalty = R.applySandbagPenalty(totals[i].allBags, 10, 100)
        totals[i].spadesTotal = totals[i].spadesTotal - penalty
        totals[i].allBags     = newAllBags
    end
    
    -- 4) derive “this hand” scores (for the right-side single-hand row)
    local liveS1 = R.spadesHand(live.team1)
    local liveH1 = R.heartsHand(live.team1)
    local liveS2 = R.spadesHand(live.team2)
    local liveH2 = R.heartsHand(live.team2)
    
    return {
        team = {
            [1] = {
                spadesScore = liveS1.handScore,
                heartsScore = liveH1.handScore,
                handScore   = (liveS1.handScore - liveH1.handScore),
                handBags    = liveS1.over,
                allBags     = totals[1].allBags,
                spadesTotal = totals[1].spadesTotal,
                heartsTotal = totals[1].heartsTotal,
                gameTotal   = totals[1].spadesTotal - totals[1].heartsTotal,
            },
            [2] = {
                spadesScore = liveS2.handScore,
                heartsScore = liveH2.handScore,
                handScore   = (liveS2.handScore - liveH2.handScore),
                handBags    = liveS2.over,
                allBags     = totals[2].allBags,
                spadesTotal = totals[2].spadesTotal,
                heartsTotal = totals[2].heartsTotal,
                gameTotal   = totals[2].spadesTotal - totals[2].heartsTotal,
            }
        }
    }
end

-- Write computed fields back into teams[*] so the UI reads a single source.
function ScoreLedger:applyToTeams()
    local snapshot = self:recompute()
    for i=1,2 do
        local src = snapshot.team[i]
        local dst = self.teams[i]
        dst.spadesScore = src.spadesScore
        dst.heartsScore = src.heartsScore
        dst.handScore   = src.handScore
        dst.handBags    = src.handBags
        dst.allBags     = src.allBags
        dst.spadesTotal = src.spadesTotal
        dst.heartsTotal = src.heartsTotal
        dst.gameTotal   = src.gameTotal
    end
end