-- IncrementingCell.lua
IncrementingCell = class()
IncrementingCell._owners = {}

function IncrementingCell:init(x, y, w, h, initialValue)
    self.x, self.y, self.w, self.h = x, y, w, h
    self.value   = initialValue or 0
    self.hasSet  = false   -- <— NEW: start as “not entered” so we draw “--”
    
    -- themeable colors (fallbacks keep existing look)
    self.colBg         = self.colBg         or Theme.cellBg
    self.colBgPressed  = self.colBgPressed  or Theme.cellBgPressed
    self.colStroke     = self.colStroke     or Theme.gridLine
    self.colText       = self.colText       or Theme.textAccentBlue
    self.colTextUnset  = self.colTextUnset  or Theme.textDisabled
    
    self.min, self.max = 0, 13
    self.wrap = true
    
-- pixels per increment while dragging (tune to taste)
self.stepPx      = 22
self.dragLastX   = nil   -- last x we sampled
self.dragAccum   = 0     -- accumulated horizontal pixels since last step
    
    self.isPressed = false
    
    self.sensor = Sensor{ parent = self }
    
    -- visual press state (like the old sensorPress:onTouch)
    self.sensor:onTouch(function(ev)
        self.isPressed = ev.state and true or false
    end)
    
    self.sensor:onDrag(function(event)
        local t = event.touch
        if t.state == BEGAN then
            self.dragLastX = t.x
            self.dragAccum = 0
            return
        end
        
        if (t.state == MOVING or t.state == ENDED) and self.dragLastX then
            -- accumulate horizontal motion
            local dx = t.x - self.dragLastX
            self.dragLastX = t.x
            self.dragAccum = self.dragAccum + dx
            
            -- how many full steps worth of motion have we accumulated?
            local steps = 0
            if self.dragAccum >=  self.stepPx then
                steps = math.floor(self.dragAccum / self.stepPx)
            elseif self.dragAccum <= -self.stepPx then
                steps = math.ceil(self.dragAccum / self.stepPx) -- negative
            end
            
            if steps ~= 0 then
                -- step once per unit so boundary logic (“--” between min/max) still applies
                local dir = (steps > 0) and 1 or -1
                for _ = 1, math.abs(steps) do
                    self:_step(dir)
                end
                -- keep only the remainder so small back-and-forth feels smooth
                self.dragAccum = self.dragAccum - steps * self.stepPx
            end
            
            if t.state == ENDED or t.state == CANCELLED then
                self.dragLastX = nil
                self.dragAccum = 0
            end
        end
    end)
    
    self.sensor:onTap(function()
        self:_step(1)
    end)
    
end
    
function IncrementingCell:draw()
    pushStyle()
    rectMode(CORNER)
    if self.isPressed then
        fill(self.colBgPressed) ; stroke(self.colStroke) ; strokeWidth(2)
    else
        fill(self.colBg) ; stroke(self.colStroke) ; strokeWidth(1)
    end
    rect(self.x, self.y, self.w, self.h)
    
    local f = self.fontSize or (self.h * 0.5)
    font("HelveticaNeue")
    fontSize(f)
    textAlign(CENTER)
    
    local label, col
    if self.hasSet then
        label = tostring(self.value)
        col   = self.colText
    else
        label = "--"
        col   = self.colTextUnset
    end
    fill(col)
    text(label, self.x + self.w/2, self.y + self.h/2)
    popStyle()
end
    
function IncrementingCell:set(v)
    if self.wrap then
        local span = self.max - self.min + 1
        v = ((v - self.min) % span + span) % span + self.min
    else
        v = math.max(self.min, math.min(self.max, v))
    end
    self.value  = v
    self.hasSet = true   -- <— mark as entered whenever we programmatically set
end

function IncrementingCell:unset()
    self.value  = 0
    self.hasSet = false
end

function IncrementingCell:inbox(t)
    return t.x >= self.x and t.x <= self.x + self.w
    and t.y >= self.y and t.y <= self.y + self.h
end

-- check/assign ownership for this touch id
function IncrementingCell:_ownsTouch(t)
    local owners = IncrementingCell._owners
    local current = owners[t.id]
    if current then return current == self end
    if t.state == BEGAN and self:inbox(t) then
        owners[t.id] = self
        return true
    end
    return false
end

-- step by +1 or -1, with "--" between max and min
function IncrementingCell:_step(delta)
    -- if we're unset, first step chooses an endpoint in the step direction
    if not self.hasSet then
        if delta > 0 then self:set(self.min) else self:set(self.max) end
        return
    end
    local v = self.value + delta
    if v > self.max or v < self.min then
        -- crossing a boundary goes to unset
        self:unset()
    else
        self:set(v)
    end
end

function IncrementingCell:touched(t)
    if not self:_ownsTouch(t) then
        return false
    end
    
    local handled = self.sensor:touched(t)
    
    if t.state == ENDED or t.state == CANCELLED then
        local owners = IncrementingCell._owners
        if owners[t.id] == self then owners[t.id] = nil end
    end
    
    return handled
end