-----------------------------------------
-- Fixed FullOrbitViewer (quaternion arcball)
-- endless rotation in all directions
-- collaboration between UberGoober, ChatGPT and DeepSeek
-----------------------------------------

FullOrbitViewer = class()

function FullOrbitViewer:init(entity, target, zoom, minZoom, maxZoom)
    self.entity   = entity
    self.camera   = entity:get(craft.camera)
    
    -- Orbit target and zoom limits
    self.target   = target or vec3(0,0,0)
    self.origin   = self.target
    self.zoom     = zoom or 5
    self.minZoom  = minZoom or 1
    self.maxZoom  = maxZoom or 20
    
    -- Input state
    self.touches  = {}
    self.prev     = {}
    
    -- Start with identity quaternion (proper initialization)
    self.q = quat(1, 0, 0, 0)
    self.entity.rotation = self.q
    
    -- Momentum (fixed calculation)
    self.spinAxis = nil         -- vec3
    self.spinVel  = 0           -- degrees per second
    self.spinDamp = 0.92
    
    -- Feel (MUCH higher sensitivity for proper movement)
    self.rotationSensitivity = 0.1  -- Increased from 0.005 (20x)
    self.panSensitivity = 0.5       -- Increased from 0.01 (50x)
    self.zoomSensitivity = 0.5      -- Increased from 0.1 (5x)
    self.momentumDamping = 0.92     -- Slightly less damping for longer momentum
    
    if touches then touches.addHandler(self, 0, true) end
    
    -- Arcball screen sphere
    -- Arcball screen sphere
    self:_updateArcballFrame()
    self.arcPrev   = nil
    
    -- Feel
    self.arcGain   = 1.0     -- 1.0 = literal great-circle; try 1.2–1.5 if you want “more movement”
    self.momentum  = vec2(0,0)
end

function FullOrbitViewer:_updateArcballFrame()
    self.arcCenter  = vec2(WIDTH * 0.5, HEIGHT * 0.5)
    self.arcRadiusX = WIDTH  * 0.5
    self.arcRadiusY = HEIGHT * 0.5
end

local function _clamp(x, lo, hi)
    if x < lo then return lo elseif x > hi then return hi else return x end
end

function FullOrbitViewer:_mapToArcball(p)
    -- Map screen point to unit hemisphere using an *elliptical* arcball
    local x =  (p.x - self.arcCenter.x) / self.arcRadiusX
    local y = -(p.y - self.arcCenter.y) / self.arcRadiusY  -- flip Y so up-swipe rotates up
    local r2 = x*x + y*y
    local z
    if r2 <= 1 then
        z = math.sqrt(1 - r2)
    else
        -- Normalize to the ellipse rim (keeps smooth motion at screen edges)
        local r = math.sqrt(r2); x, y = x/r, y/r; z = 0
    end
    return vec3(x, y, z)
end

-- Project a 2D screen point to a point z units away along the view ray
function FullOrbitViewer:project(p, z)
    local origin, dir = self.camera:screenToRay(p)
    return origin + dir * z
end

-- Smooth overscroll curve for zooming
local function scrollDamping(x, s)
    return s * math.log(x + s) - s * math.log(s)
end

function FullOrbitViewer:_rotateByDeltas(dx, dy)
    -- Fixed: Use proper arcball rotation
    -- Yaw about world-up (vertical axis)
    local yawQ = quat.angleAxis(-dx * self.rotationSensitivity, vec3(0, 1, 0))
    
    -- Pitch about camera's local right axis (horizontal axis)
    local right = self.entity.right
    if right:len() < 1e-6 then right = vec3(1, 0, 0) end
    
    local pitchQ = quat.angleAxis(-dy * self.rotationSensitivity, right)
    
    -- Fixed: Correct composition order - apply pitch first, then yaw
    -- This gives more intuitive rotation behavior
    self.q = yawQ * (pitchQ * self.q)
    self.q:normalize() -- Ensure it stays a unit quaternion
end

function FullOrbitViewer:update()
    self:_updateArcballFrame()
    -- Apply momentum when no touches
    -- in update(), before applying entity transform
    if #self.touches == 0 and not self.capturedScroll then
        local m = self.momentum
        if m:len() > 0.1 then
            -- synthesize a tiny step from the screen center by that momentum
            local p0 = self.arcCenter + vec2(self.momentum.x, self.momentum.y) * DeltaTime * 60
            local p1 = self.arcCenter
            self:_arcballRotate(p0, p1)
            self.momentum = self.momentum * self.momentumDamping
            if self.momentum:len() < 0.1 then self.momentum = vec2(0,0) end
        end
        -- your soft zoom limit settling can stay here        
        -- Smooth zoom limits
        if self.zoom > self.maxZoom then
            self.zoom = self.zoom + (self.maxZoom - self.zoom) * 0.2
        elseif self.zoom < self.minZoom then
            self.zoom = self.zoom + (self.minZoom - self.zoom) * 0.2
        end
    end
    
    -- Two-finger pan + pinch
    if #self.touches == 2 then
        local mid = self:pinchMid()
        local dist = self:pinchDist()
        
        if not self.prev.mid then
            self.prev.mid = mid
            self.prev.dist = dist
            self.prev.target = vec3(self.target:unpack())
            self.prev.zoom = self.zoom
        end
        
        -- Panning
        local p1 = self:project(self.prev.mid, self.zoom)
        local p2 = self:project(mid, self.zoom)
        self.target = self.prev.target + (p1 - p2) * self.panSensitivity
        
        -- Pinch to zoom
        local zoomFactor = self.prev.dist / math.max(1, dist)
        self.zoom = self:clampZoom(self.prev.zoom * zoomFactor)
    else
        self.prev.mid = nil
        self.prev.dist = nil
    end
    
    -- Apply final transform
    self.entity.rotation = self.q
    self.entity.position = self.target - self.entity.forward * self.zoom
end

-- Pinch helpers
function FullOrbitViewer:pinchDist()
    local p1 = vec2(self.touches[1].x, self.touches[1].y)
    local p2 = vec2(self.touches[2].x, self.touches[2].y)
    return p1:dist(p2)
end

function FullOrbitViewer:pinchMid()
    local p1 = vec2(self.touches[1].x, self.touches[1].y)
    local p2 = vec2(self.touches[2].x, self.touches[2].y)
    return (p1 + p2) * 0.5
end

-- Keep zoom within limits
function FullOrbitViewer:clampZoom(zoom)
    if zoom > self.maxZoom then
        local overshoot = zoom - self.maxZoom
        overshoot = scrollDamping(overshoot, 5.0)
        zoom = self.maxZoom + overshoot
    elseif zoom < self.minZoom then
        local overshoot = self.minZoom - zoom
        overshoot = scrollDamping(overshoot, 5.0)
        zoom = self.minZoom - overshoot
    end
    return zoom
end

-- Panning
function FullOrbitViewer:pan(p1, p2)
    local a = self:project(p1, self.zoom)
    local b = self:project(p2, self.zoom)
    self.target = self.target + (a - b) * self.panSensitivity
end

-- Scroll gestures (fixed)
function FullOrbitViewer:scroll(gesture)
    local panMode = gesture.shift
    local zoomMode = gesture.alt
    
    if gesture.state == BEGAN then
        if #self.touches > 0 then return false end
        self.capturedScroll = true
        self.prev.zoom = self.zoom
        self.prev.mid = gesture.location
        return true
        
    elseif gesture.state == MOVING then
        if panMode then
            self:pan(gesture.location - gesture.delta, gesture.location)
        elseif zoomMode then
            local zoomDelta = (gesture.location.y - self.prev.mid.y) * self.zoomSensitivity
            self.zoom = self:clampZoom(self.prev.zoom - zoomDelta)
        else
            -- Fixed: Consistent rotation direction with proper sensitivity
            self:_rotateByDeltas(gesture.delta.x, gesture.delta.y)
        end
        self.prevGestureDelta = gesture.delta
        
    elseif gesture.state == ENDED or gesture.state == CANCELLED then
        self.capturedScroll = false
        if not panMode and not zoomMode and self.prevGestureDelta then
            -- Fixed: Proper momentum calculation with higher sensitivity
            self.momentum = vec2(
            self.prevGestureDelta.x * self.rotationSensitivity * 5,
            self.prevGestureDelta.y * self.rotationSensitivity * 5
            )
        end
    end
end

-- Touch handling (fixed)
function FullOrbitViewer:touched(touch)
    self:_updateArcballFrame()
    -- Double tap to recenter
    if touch.tapCount == 2 then
        self.target = self.origin
        self.q = quat(1, 0, 0, 0) -- Reset rotation
        self.momentum = vec2(0, 0)
        return true
    end
    
    if self.capturedScroll then return false end
    
    -- in touched(touch)
    if touch.state == BEGAN and #self.touches < 2 then
        table.insert(self.touches, touch)
        self.arcPrev = vec2(touch.x, touch.y)
        return true
        
    elseif touch.state == MOVING then
        for i=1,#self.touches do
            if self.touches[i].id == touch.id then self.touches[i] = touch; break end
        end
        if #self.touches == 1 then
            local p1 = vec2(touch.x, touch.y)
            if self.arcPrev then self:_arcballRotate(p1, self.arcPrev) end
            -- keep some velocity for fling
            self.momentum = vec2(touch.deltaX, touch.deltaY)
            self.arcPrev = p1
        else
            self.arcPrev = nil  -- reset when we go multi-touch
        end
        
    elseif touch.state == ENDED or touch.state == CANCELLED then
        for i=#self.touches,1,-1 do
            if self.touches[i].id == touch.id then table.remove(self.touches, i); break end
        end
        if #self.touches == 0 then
            -- convert last pixel/s into momentum we’ll arcball-apply in update()
            local d = self.momentum
            if d:len() < 2 then d = vec2(0,0) end
            self.momentum = d
            self.arcPrev  = nil
        end
    end
    return false
end
    
-- Rotate camera by dragging from p0 to p1 in screen space (arcball)
function FullOrbitViewer:_arcballRotate(p0, p1)
    local v0 = self:_mapToArcball(p0)
    local v1 = self:_mapToArcball(p1)
    
    local dot  = _clamp(v0:dot(v1), -1, 1)
    local axis = v0:cross(v1)
    local len  = axis:len()
    if len < 1e-6 then return end
    
    axis = axis / len
    -- Codea quat.angleAxis expects DEGREES
    local angleDeg = math.deg(math.acos(dot)) * self.arcGain
    local dq = quat.angleAxis(angleDeg, axis)
    
    -- Compose: new rotation = delta * current
    self.q = self.q * dq
    self.q:normalize()
end

function FullOrbitViewer:_rotateByDeltas(dx, dy)
    local yawQ   = quat.angleAxis( dx * self.rotationSensitivity, vec3(0,1,0))
    local right  = self.entity.right; if right:len() < 1e-6 then right = vec3(1,0,0) end
    local pitchQ = quat.angleAxis( dy * self.rotationSensitivity, right)
    self.q = self.q * yawQ * pitchQ
    self.q:normalize()
end
