HEX_DIRS = { -- assume top is flat
  {q= 0, r=-1}, -- 1 north
  {q=-1, r= 0}, -- 2 northwest
  {q=-1, r= 1}, -- 3 southwest
  {q= 0, r= 1}, -- 4 south
  {q= 1, r= 0}, -- 5 southeast
  {q= 1, r=-1}, -- 6 northeast
}

function sync3DTilesFromBoard(board)
  -- mark all existing 3D tiles unused
  local seen = {}
  
  for _,cell in ipairs(board:allHexCellsAsFlatList()) do
    local q, r = cell.q, cell.r
    local key = keyQR(q,r)
    seen[key] = true
    
    local ent = Game.tiles3D[key]
    
    if not ent then
      -- create 3D tile from the TileModel
      ent = createHexTileFromTileModel(cell.tile)
      ent.parent = boardRoot
      ent.position = coordsOfHexPosition(q, r, Game.radius)
      Game.tiles3D[key] = ent
    else
      -- always rebind source of truth
      ent.model2D = cell.tile
      
      -- 🔍 compare logical vs drawn state
      if not sidesEqual(ent._drawnSides, cell.tile.sides) then
        refreshTileEntityFromModel(ent)
      end
      
      ent.position = coordsOfHexPosition(q, r, Game.radius)
    end
  end
  
  
  -- remove any 3D tiles whose board tile no longer exists
  for key,ent in pairs(Game.tiles3D) do
    if not seen[key] then
      ent:destroy()
      Game.tiles3D[key] = nil
    end
  end
end

function rotate3D(dir, tile3D)
  local model = tile3D.model2D
  assert(model, "rotate3D: tile has no model2D")
  
  local delta = -60
  local tweenY = dir == "left" and delta - 60 or 0
  
  tile3D.eulerAngles = vec3(0, delta, 0)
    
    tween(0.11, tile3D,
    { eulerAngles = vec3(0, tweenY, 0) },
    nil,
    function()        
      local endPositionDirection = dir == "left" and "right" or "left"
      model:rotate(endPositionDirection)     
       
      tile3D.eulerAngles = vec3(0, delta, 0)        refreshTileEntityFromModel(tile3D)
    end)
end

function copySides(sides)
  local t = {}
  for i = 1, 6 do t[i] = sides[i] end
  return t
end

function sidesEqual(a, b)
  if not a or not b then return false end
  for i = 1, 6 do
    if a[i] ~= b[i] then return false end
  end
  return true
end

function createHexTile(colorNames)
  local colors = colorsFromSideNames(colorNames)
  
  if not hexTileModel then
    hexTileModel = craft.model()
    
    local verts = {}
    local norms = {}
    local cols  = {}
    local uvs   = {}
    local idx   = {}
    
    local r = 1
    local h = r / 3
    local topY = h * 0.5
    local botY = -h * 0.5
    
    local ringTop = {}
    local ringBot = {}
    
    for i = 0, 5 do
      local a = math.rad(i * 60)
      local x = r * math.cos(a)
      local z = r * math.sin(a)
      ringTop[i+1] = vec3(x, topY, z)
      ringBot[i+1] = vec3(x, botY, z)
    end
    
    ------------------------------------------------
    -- TOP (use your working top builder)
    ------------------------------------------------
    local tp, tn, tu, ti =
    makeSteppedHexTopTables(
    r,
    topY,
    TOP_RAISE_Y,
    INNER_HEX_FRAC
    )
    
    local baseTop = #verts
    for i,v in ipairs(tp) do
      table.insert(verts, v)
      table.insert(norms, tn[i])
      table.insert(cols, color(255))   -- keep white so texture is not tinted
      table.insert(uvs,  tu[i])
    end
    for _,k in ipairs(ti) do
      table.insert(idx, baseTop + k)
    end
    
    ------------------------------------------------
    -- BOTTOM (optional; keep simple)
    ------------------------------------------------
    table.insert(verts, vec3(0, botY, 0))
    table.insert(norms, vec3(0,-1,0))
    table.insert(cols, color(255))
    table.insert(uvs, vec2(0.5,0.5))
    local bc = #verts
    
    for i=1,6 do
      table.insert(verts, ringBot[i])
      table.insert(norms, vec3(0,-1,0))
      table.insert(cols, color(255))
      table.insert(uvs, vec2(0.5,0.5))
    end
    
    for i=1,6 do
      table.insert(idx, bc)
      table.insert(idx, bc+((i%6)+1))
      table.insert(idx, bc+i)
    end
    
    ------------------------------------------------
    -- SIDES (THIS IS THE FIX)
    -- Give each side a UV that samples inside the matching wedge color
    ------------------------------------------------
    for i=1,6 do
      local ni = (i%6)+1
      
      local v1 = ringTop[i]
      local v2 = ringTop[ni]
      local v3 = ringBot[ni]
      local v4 = ringBot[i]
      
      local baseSide = #verts+1
      
      table.insert(verts, v1)
      table.insert(verts, v2)
      table.insert(verts, v3)
      table.insert(verts, v4)
      
      local mid = (v1+v2)*0.5
      local n = vec3(mid.x,0,mid.z):normalize()
      
      -- Pick a UV point safely inside wedge i
      -- Use mid-angle between vertex i and i+1:
      local aMid = math.rad((i-0.5) * 60)
      
      -- uvRadius should be inside your ring area, not near the edge.
      -- 0.30 is safe given your mask uses R=size*0.45 and center at 0.5.
      local uvRadius = 0.30
      local u = 0.5 + uvRadius * math.cos(aMid)
      local v = 0.5 + uvRadius * math.sin(aMid)
      local uvSide = vec2(u, v)
      
      for k=1,4 do
        table.insert(norms, n)
        table.insert(cols, color(255))   -- keep white; texture provides color
        table.insert(uvs, uvSide)        -- THIS prevents black and matches the wedge
      end
      
      table.insert(idx, baseSide)
      table.insert(idx, baseSide+2)
      table.insert(idx, baseSide+1)
      
      table.insert(idx, baseSide)
      table.insert(idx, baseSide+3)
      table.insert(idx, baseSide+2)
    end
    
    hexTileModel.positions = verts
    hexTileModel.normals   = norms
    hexTileModel.colors    = cols
    hexTileModel.uvs       = uvs
    hexTileModel.indices   = idx
    
  end -- model cache
  
  -- build texture from the SAME color table
  local tex = standardHexTexture(colors)
  
  
  local e = scene:entity()
  e.model = hexTileModel
  local mat = craft.material(asset.builtin.Materials.Standard)
  mat.diffuse = color(255)  -- must be white
  mat.map = tex
  e.material = mat
  e.colorNames = colorNames -- length must be 6
  e.rot = 0
  e.eulerAngles = vec3(0, rotToYaw(e.rot), 0)
  e.model2D = TileModel("X", colorNames, "normal")
  assertValidTile(e)
  return e
end

-- returns: positions, normals, uvs, indices
function makeHexTopTables(radius, y)
  local positions = {}
  local normals   = {}
  local uvs       = {}
  local indices   = {}
  
  y = y or 0
  
  -- center
  positions[1] = vec3(0, y, 0)
  normals[1]   = vec3(0, 1, 0)
  uvs[1]       = vec2(0.5, 0.5)
  
  -- ring
  for i=0,5 do
    local a = math.rad(i * 60)
    local x = radius * math.cos(a)
    local z = radius * math.sin(a)
    
    local vi = i + 2
    positions[vi] = vec3(x, y, z)
    normals[vi]   = vec3(0, 1, 0)
    
    -- EXACT working UV math
    uvs[vi] = vec2(
    (x / radius + 1) * 0.5,
    (z / radius + 1) * 0.5
    )
  end
  
  -- fan indices
  for i=2,7 do
    local n = (i == 7) and 2 or (i + 1)
    table.insert(indices, 1)
    table.insert(indices, i)
    table.insert(indices, n)
  end
  
  return positions, normals, uvs, indices
end

 TOP_RAISE_Y     = 0.2    -- height of inner hex
 INNER_HEX_FRAC  = 0.82    -- radius fraction vs outer hex
function makeSteppedHexTopTables(radius, baseY, raiseY, innerFrac)
  local positions = {}
  local normals   = {}
  local uvs       = {}
  local indices   = {}
  
  local innerR = radius * innerFrac
  local y0 = baseY
  local y1 = baseY + raiseY
  
  ----------------------------------------------------------------
  -- Build outer ring (flat, same as before)
  ----------------------------------------------------------------
  local outer = {}
  for i=0,5 do
    local a = math.rad(i * 60)
    outer[i+1] = vec3(
    radius * math.cos(a),
    y0,
    radius * math.sin(a)
    )
  end
  
  ----------------------------------------------------------------
  -- Build inner raised hex
  ----------------------------------------------------------------
  local inner = {}
  for i=0,5 do
    local a = math.rad(i * 60)
    inner[i+1] = vec3(
    innerR * math.cos(a),
    y1,
    innerR * math.sin(a)
    )
  end
  
  ----------------------------------------------------------------
  -- Center vertex (inner top)
  ----------------------------------------------------------------
  positions[1] = vec3(0, y1, 0)
  normals[1]   = vec3(0,1,0)
  uvs[1]       = vec2(0.5,0.5)
  
  -- inner ring vertices
  for i=1,6 do
    positions[#positions+1] = inner[i]
    normals[#normals+1]     = vec3(0,1,0)
    uvs[#uvs+1] = vec2(
    (inner[i].x / radius + 1) * 0.5,
    (inner[i].z / radius + 1) * 0.5
    )
  end
  
  -- inner top fan
  for i=2,7 do
    local n = (i == 7) and 2 or (i + 1)
    indices[#indices+1] = 1
    indices[#indices+1] = i
    indices[#indices+1] = n
  end
  
  ----------------------------------------------------------------
  -- Slanted ring between inner and outer hex
  ----------------------------------------------------------------
  for i=1,6 do
    local ni = (i % 6) + 1
    
    local a = inner[i]
    local b = inner[ni]
    local c = outer[ni]
    local d = outer[i]
    
    local base = #positions + 1
    
    positions[base]     = a
    positions[base + 1] = b
    positions[base + 2] = c
    positions[base + 3] = d
    
    -- compute face normal
    local nrm = ((d - a):cross(b - a)):normalize()
    
    -- force bevel normals to face upward + outward
    if nrm.y < 0 then
      nrm = -nrm
    end
    
    nrm.y = math.max(nrm.y, 0.45)
    nrm = nrm:normalize()
    
    for k=1,4 do
      normals[#normals+1] = nrm
      uvs[#uvs+1] = vec2(
      (positions[base+k-1].x / radius + 1) * 0.5,
      (positions[base+k-1].z / radius + 1) * 0.5
      )
    end
    
    -- two triangles
    indices[#indices+1] = base
    indices[#indices+1] = base + 2
    indices[#indices+1] = base + 1
    
    indices[#indices+1] = base
    indices[#indices+1] = base + 3
    indices[#indices+1] = base + 2
  end
  
  return positions, normals, uvs, indices
end

function createHexTileFromTileModel(tileModel)
  local e = createHexTile(tileModel.sides)
  e.model2D = tileModel
  return e
end

function refreshTileEntityFromModel(ent)
  local model = ent.model2D
  assert(model, "3D tile has no model2D")
  
  -- rebuild texture from model state
  local colors = colorsFromSideNames(model.sides)
  ent.material.map = standardHexTexture(colors)
  
  -- 🔒 record exactly what we drew
  ent._drawnSides = copySides(model.sides)
end


function newRandomHexAt(hexPosition)
  local q = hexPosition.q
  local r = hexPosition.r
  
  local p = coordsOfHexPosition(q, r, 1)
  
  local e = createHexTile(randomSideColors())
  e.position = p
  e.rot = 0
  e.eulerAngles = vec3(0, rotToYaw(e.rot), 0)
  e._q = q
  e._r = r
  
  return e
end

function assertValidTile(ent)
  assert(ent.model2D and #ent.model2D.sides == 6,
  "Tile must have exactly 6 sides")
end