function makeHexMaskTexture(size, innerFrac, edgeColors, centerCol)
  local img = image(size, size)
  
  centerCol = centerCol or CENTER_COLOR_3D or color(0)
  
  ------------------------------------------------
  -- Build hex vertices
  ------------------------------------------------
  local R = size * 0.5
  local verts = {}
  
  for i=0,5 do
    local a = math.rad(i*60)
    verts[i+1] = vec2(
    R * math.cos(a),
    R * math.sin(a)
    )
  end
  
  ------------------------------------------------
  -- Build OUTER RING mesh (6 wedges)
  ------------------------------------------------
  local ring = mesh()
  ring:resize(18) -- 6 triangles
  
  local idx = 1
  for i=1,6 do
    local v1 = verts[i]
    local v2 = verts[(i%6)+1]
    
    local base = (#edgeColors == 7) and 1 or 0   -- 7 means [1]=center, [2..7]=edges
    local col  = edgeColors[i + base] or color(255)
    
    ring:vertex(idx, vec2(0,0))
    ring:color(idx, col.r, col.g, col.b, col.a)
    idx = idx+1
    
    ring:vertex(idx, v1)
    ring:color(idx, col.r, col.g, col.b, col.a)
    idx = idx+1
    
    ring:vertex(idx, v2)
    ring:color(idx, col.r, col.g, col.b, col.a)
    idx = idx+1
  end
  
  ------------------------------------------------
  -- Build STAR inner hex (concave sides)
  ------------------------------------------------
  local inner = mesh()
  
  local baseR   = R * innerFrac
  local bulgeR  = baseR * 0.08   -- controls curve depth
  local arcSteps = 30             -- smoothness
  
  local verts2 = {}
  local cols2  = {}
  
  for i=1,6 do
    local a1 = math.rad((i-1)*60)
    local a2 = math.rad(i*60)
    
    -- corner points
    local p1 = vec2(baseR*math.cos(a1), baseR*math.sin(a1))
    local p2 = vec2(baseR*math.cos(a2), baseR*math.sin(a2))
    
    -- mid-angle for inward bulge
    local am = (a1+a2)*0.5
    local mid = vec2(
    bulgeR*math.cos(am),
    bulgeR*math.sin(am)
    )
    
    -- build arc from p1 -> mid -> p2
    local arc = {}
    for s=0,arcSteps do
      local t = s/arcSteps
      
      -- quadratic bezier
      local a = p1*(1-t)*(1-t) +
      mid*2*(1-t)*t +
      p2*t*t
      
      arc[#arc+1] = a
    end
    
    -- triangulate fan from center
    for s=1,#arc-1 do
      verts2[#verts2+1] = vec2(0,0)
      verts2[#verts2+1] = arc[s]
      verts2[#verts2+1] = arc[s+1]
      
      for k=1,3 do
        cols2[#cols2+1] = centerCol
      end
    end
  end
  
  inner:resize(#verts2)
  for i,v in ipairs(verts2) do
    inner:vertex(i,v)
    local c = cols2[i]
    inner:color(i,c.r,c.g,c.b,c.a)
  end
  
  ------------------------------------------------
  -- Render to texture
  ------------------------------------------------
  setContext(img)
  pushMatrix()
  pushStyle()
  
  background(0,0,0,0)
  translate(size/2, size/2)
  
  ring:draw()
  inner:draw()
  ------------------------------------------------
  -- Thin outline
  ------------------------------------------------
  stroke(177, 105)
  strokeWidth(2)
  noFill()
  
  local inset = R * 0.99  -- keeps it just inside the edge
  
  local outline = {}
  for i=0,5 do
    local a = math.rad(i*60)
    outline[i+1] = vec2(
    inset * math.cos(a),
    inset * math.sin(a)
    )
  end
  
  for i=1,6 do
    local p1 = outline[i]
    local p2 = outline[(i%6)+1]
    line(p1.x, p1.y, p2.x, p2.y)
  end
  popStyle()
  popMatrix()
  setContext()
  
  return img
end

function standardHexTexture(edgeColors)
  return makeHexMaskTexture(256, 0.85, edgeColors, CENTER_COLOR_3D)
end