Game = {
  radius = 1,          -- world radius used by coordsOfHexPosition()
  bag = {},
  bagIndex = 1,
  board = {},          -- key -> tile
  hover = {q=0,r=0},
  haveHover = false,
  previewEnt = nil,
  highlightEnt = nil,
  confirmMode = false,
  awaitingConfirm = false,
  tiles3D = {},
  activeSeat = 1
}

Local = {
  seat = 1   -- this device is seat 1 or 2
}

function gameInit()  
  Game.highlightEnt = highlightHexAt({q=0,r=0})
  Game.highlightEnt.active = false
end

function newGame()
  -- stop async state
  Game.aiThinking = false
  Game.aiThinkUntil = nil
  
  -- reset core state
  currentAvailableColors = randomUniqueColors(2)
  board:reset()
  match:reset()
  
  -- clear 3D entities
  for k,e in pairs(Game.tiles3D) do
    if e and e.destroy then e:destroy() end
    Game.tiles3D[k] = nil
  end
  
  -- rebuild visuals from board
  sync3DTilesFromBoard(board)
  highlighPlayableHexes()
  ui3D:refreshPreviewFrom(board)
end

function moveCandidate3D(q, r)
  board:moveCandidate(q, r)
  sync3DTilesFromBoard(board)
  highlighPlayableHexes()
end

function mySeat()
  return Local.seat
end

function opponentSeat()
  return 3 - Local.seat
end

function myScore()
  return match.seats[Local.seat].score
end

function opponentScore()
  return match.seats[3 - Local.seat].score
end

-- plays EXACTLY ONE turn using an explicit model at q,r
-- (replicates commitCandidate3D steps 2..5, no extras)
function commitTileWithModel(model, q, r)
  assert(model and q and r)
  
  -- 2) score
  local points = board:tileMatchesAt(model, q, r)
  match:addScore(points)
  
  -- 3) commit model to board
  board:commitScoringModel(model, q, r)
  
  -- 4) reflect board state
  sync3DTilesFromBoard(board)
  highlighPlayableHexes()
  ui3D:refreshPreviewFrom(board)
  
  return points
end

-- rename commitCandidate3D -> runLocalPlayerTurn
function runLocalPlayerTurn()
  -- 1) get model + position for scoring
  local model, q, r = board:cloneOfScoringModel()
  assert(model and q and r)
  
  commitTileWithModel(model, q, r)
  
  match:endTurn()
  
  if match.gameEnded then
    return
  end
  
  -- if we're using GC for turn exchange, don't also run local AI
  if gc and gc.endTurn and not LOCAL_AI_ENABLED then
    gc:endTurn()
  end
  
  -- local AI mode only
  if LOCAL_AI_ENABLED and (not match:isLocalTurn()) then
    Game.aiThinking = true
    Game.aiThinkUntil = ElapsedTime + math.random(8) * 0.1
  end
end

-- AI uses the SAME pipeline: compute move -> build model -> commitTileWithModel
function runAITurn()
  local move = computeBestAIMove(board)
  
  -- no legal moves: treat as a "pass" so you don't get stuck on AI's seat forever
  if not move.q then
    match:endTurn()
    ui3D:refreshPreviewFrom(board)
    return
  end
  
  local base = board:pendingCandidateModel()
  assert(base)
  
  local model = base:clone()
  for i = 1, move.rot do
    model:rotate("left")
  end
  
  commitTileWithModel(model, move.q, move.r)
  
  match:endTurn()
end

function computeBestAIMove(board)
  local best = { q=nil, r=nil, rot=0, score=-1 }
  
  local baseModel = select(1, board:cloneOfScoringModel())
  assert(baseModel)
  
  for _,pos in ipairs(board:playableHexes()) do
    for rot=0,5 do
      local testModel = baseModel:clone()
      for i=1,rot do testModel:rotate("left") end
      
      local score = board:tileMatchesAt(testModel, pos.q, pos.r)
      
      if score > best.score then
        best = { q=pos.q, r=pos.r, rot=rot, score=score }
      end
    end
  end
  
  return best
end