UITweaks = {
  
  textColor   = color(80, 100, 39),
  
  glowColor   = color(161, 152, 28),
  
  labelColor  = color(68, 77, 31, 204),
  
  textAlpha   = 255,
  
  glowAlpha   = 15,
  
  labelAlpha  = 102,
  
  fontMain    = 64,
  
  fontLabel   = 16,
  
  gapBetweenWordAndNumber = 57,
  
  numberYOffset = 0,
  
  bobSpeed    = 2.500,
  
  bobAmount   = 0.050
  
}

function setupUITweaks()
  parameter.color("ui_textColor", UITweaks.textColor,
  function(c) UITweaks.textColor = c end)
  
  parameter.color("ui_glowColor", UITweaks.glowColor,
  function(c) UITweaks.glowColor = c end)
  
  parameter.color("ui_labelColor", UITweaks.labelColor,
  function(c) UITweaks.labelColor = c end)
  
  parameter.integer("ui_textAlpha", 0, 255, UITweaks.textAlpha,
  function(v) UITweaks.textAlpha = v end)
  
  parameter.integer("ui_glowAlpha", 0, 255, UITweaks.glowAlpha,
  function(v) UITweaks.glowAlpha = v end)
  
  parameter.integer("ui_labelAlpha", 0, 255, UITweaks.labelAlpha,
  function(v) UITweaks.labelAlpha = v end)
  
  parameter.integer("ui_fontMain", 24, 120, UITweaks.fontMain,
  function(v) UITweaks.fontMain = v end)
  
  parameter.integer("ui_fontLabel", 10, 48, UITweaks.fontLabel,
  function(v) UITweaks.fontLabel = v end)
  
  parameter.integer("ui_labelGap", -100, 100, UITweaks.gapBetweenWordAndNumber,
  function(v) UITweaks.gapBetweenWordAndNumber = v end)
  
  parameter.integer("ui_numberYOffset", -100, 100, UITweaks.numberYOffset,
  function(v) UITweaks.numberYOffset = v end)
  
  parameter.number("ui_bobSpeed", 0, 6, UITweaks.bobSpeed,
  function(v) UITweaks.bobSpeed = v end)
  
  parameter.number("ui_bobAmount", 0, 0.3, UITweaks.bobAmount,
  function(v) UITweaks.bobAmount = v end)
  
  parameter.action("outputSettings", function()
    local t = UITweaks
    
    print("UITweaks = {")
    print(string.format("  textColor   = color(%d, %d, %d),",
    t.textColor.r, t.textColor.g, t.textColor.b))
    print(string.format("  glowColor   = color(%d, %d, %d),",
    t.glowColor.r, t.glowColor.g, t.glowColor.b))
    print(string.format("  labelColor  = color(%d, %d, %d),",
    t.labelColor.r, t.labelColor.g, t.labelColor.b))
    print(string.format("  textAlpha   = %d,", t.textAlpha))
    print(string.format("  glowAlpha   = %d,", t.glowAlpha))
    print(string.format("  labelAlpha  = %d,", t.labelAlpha))
    print(string.format("  fontMain    = %d,", t.fontMain))
    print(string.format("  fontLabel   = %d,", t.fontLabel))
    print(string.format(
    "  gapBetweenWordAndNumber = %d,",
    t.gapBetweenWordAndNumber))
    print(string.format("  numberYOffset = %d,", t.numberYOffset))
    print(string.format("  bobSpeed    = %.3f,", t.bobSpeed))
    print(string.format("  bobAmount   = %.3f", t.bobAmount))
    print("}")
  end)
end




function drawCandidateMatchCount()
  local cell = board:getCandidateCell()
  if not cell then return end
  
  local e = Game.tiles3D[keyQR(cell.q, cell.r)]
  if not e or not e.active then return end
  
  local cam = scene.camera:get(craft.camera)
  if not cam then return end
  
  local matches = board:candidateMatches()
  
  -- hover point a bit above tile
  local wp = e.position + vec3(0, 0.55 + math.sin(ElapsedTime*3)*0.06, 0)
  local sp = cam:worldToScreen(wp)
  
  pushStyle()
  textAlign(CENTER)
  fontSize(72)
  
  -- glow (draw text multiple times slightly offset)
  fill(95, 184, 42, 120)
  for ox=-2,2 do
    for oy=-2,2 do
      text(tostring(matches), sp.x + ox, sp.y + oy)
    end
  end
  
  -- soft glow halo
  fill(137, 223, 63, 70)
  
  for i=1,14 do
    local a = i/14 * math.pi*2
    local r = 6
    local ox = math.cos(a) * r
    local oy = math.sin(a) * r
    text(tostring(matches), sp.x + ox, sp.y + oy)
  end
  
  -- core
  fill(255, 217)
  text(tostring(matches), sp.x, sp.y)
  popStyle()
end

function drawOpponentLastMovePoints()

  local last = board.lastCommit
  if not last or not last.points or last.points <= 0 then return end
  
  local e = Game.tiles3D[keyQR(last.q, last.r)]
  if not e or not e.active then return end
  
  local cam = scene.camera:get(craft.camera)
  if not cam then return end
  
  local T = UITweaks
  
  local wp = e.position +
  vec3(
  0,
  0.65 + math.sin(ElapsedTime * T.bobSpeed) * T.bobAmount,
  0
  )
  
  local sp = cam:worldToScreen(wp)
  
  pushStyle()
  textAlign(CENTER)
  
  ----------------------------------------------------------------
  -- label
  ----------------------------------------------------------------
  font("AvenirNext-HeavyItalic")
  fontSize(T.fontLabel)
  fill(T.labelColor)
  text("opponent", sp.x, sp.y + T.gapBetweenWordAndNumber)
  text("scored", sp.x, sp.y + T.gapBetweenWordAndNumber - 18)
  ----------------------------------------------------------------
  -- number
  ----------------------------------------------------------------
  fontSize(T.fontMain)
  local txt = tostring(last.points)
  local ny = sp.y + T.numberYOffset
  
  -- glow pass
  fill(T.glowColor.r, T.glowColor.g, T.glowColor.b, T.glowAlpha)
  for ox = -2, 2 do
    for oy = -2, 2 do
      text(txt, sp.x + ox, ny + oy)
    end
  end
  
  -- halo
  for i = 1, 12 do
    local a = i / 12 * math.pi * 2
    local r = 6
    text(
    txt,
    sp.x + math.cos(a) * r,
    ny + math.sin(a) * r
    )
  end
  
  -- core
  fill(T.textColor.r, T.textColor.g, T.textColor.b, T.textAlpha)
  text(txt, sp.x, ny)
  
  popStyle()
end

local function ensureOpponentGlow()
  if OpponentGlow and OpponentGlow.material then
    return
  end
  
  -- 🔥 hard reset (entity was destroyed or invalid)
  if OpponentGlow then
    OpponentGlow:destroy()
  end
  
  OpponentGlow = scene:entity()
  OpponentGlow.model = createGlowRing()
  OpponentGlow._pulseGen = 0
  
  local m = craft.material(asset.builtin.Materials.Basic)
  m.diffuse = color(0, 255, 0)
  m.opacity = 0
  m.blendMode = ADDITIVE
  m.renderQueue = TRANSPARENT
  
  OpponentGlow.material = m
end

function drawOpponentMoveLabel()
  local last = board.lastCommit
  if not last then return end
  
  local ent = Game.tiles3D[keyQR(last.q, last.r)]
  if not ent then return end
  
  local cam = scene.camera:get(craft.camera)
  if not cam then return end
  
  -- world position slightly above tile
  local wp = ent.position + vec3(0, 0.6, 0)
  local sp = cam:worldToScreen(wp)
  
  pushStyle()
  textAlign(CENTER)
  fontSize(16)
  fill(255)
  font("HelveticaNeue-Bold")
  text("opponent’s\nmove", sp.x, sp.y - 15)
  
  popStyle()
end

function createGlowRing()
  local m = craft.model()
  
  local verts, norms, cols, idx = {}, {}, {}, {}
  local r1 = 0.95
  local r2 = 1.15
  local y  = 0.04
  
  for i = 0,5 do
    local a1 = math.rad(i * 60)
    local a2 = math.rad((i + 1) * 60)
    
    local p1 = vec3(math.cos(a1) * r1, y, math.sin(a1) * r1)
    local p2 = vec3(math.cos(a1) * r2, y, math.sin(a1) * r2)
    local p3 = vec3(math.cos(a2) * r2, y, math.sin(a2) * r2)
    local p4 = vec3(math.cos(a2) * r1, y, math.sin(a2) * r1)
    
    local base = #verts + 1
    
    verts[#verts+1] = p1
    verts[#verts+1] = p2
    verts[#verts+1] = p3
    verts[#verts+1] = p4
    
    for _ = 1,4 do
      norms[#norms+1] = vec3(0,1,0)
      cols[#cols+1]  = color(80,255,120,180)
    end
    
    idx[#idx+1] = base
    idx[#idx+1] = base+1
    idx[#idx+1] = base+2
    
    idx[#idx+1] = base
    idx[#idx+1] = base+2
    idx[#idx+1] = base+3
  end
  
  m.positions = verts
  m.normals   = norms
  m.colors    = cols
  m.indices   = idx
  m.blend     = true
  
  return m
end

function spawnOpponentGlowAtTile(tileEnt)
  ensureOpponentGlow()
  -- 🔒 new generation
  OpponentGlow._pulseGen = OpponentGlow._pulseGen + 1
  local myGen = OpponentGlow._pulseGen
  
  -- knobs (tweak freely)
  local pulseUpTime   = 0.95
  local pulseDownTime = 0.95
  local scaleMin      = 0.75
  local scaleMax      = 0.85
  local opacityMin    = 0.05
  local opacityMax    = 0.9
  
  -- reparent & reset
  OpponentGlow.parent = tileEnt
  OpponentGlow.position = vec3(0, 0.2, 0)
  OpponentGlow.eulerAngles = vec3(0, 0, 0)
  OpponentGlow.active = true
  
  OpponentGlow.scale = vec3(scaleMin, scaleMin, scaleMin)
  OpponentGlow.material.opacity = opacityMin
  
  local function pulse()
    if not opponentGlowAlive(myGen) then return end
    
    tween(pulseUpTime, OpponentGlow.material,
    { opacity = opacityMax },
    tween.easing.quadOut
    )
    
    tween(pulseUpTime, OpponentGlow,
    { scale = vec3(scaleMax, scaleMax, scaleMax) },
    tween.easing.quadOut,
    function()
      if not opponentGlowAlive(myGen) then return end
      
      tween(pulseDownTime, OpponentGlow.material,
      { opacity = opacityMin },
      tween.easing.quadIn
      )
      
      tween(pulseDownTime, OpponentGlow,
      { scale = vec3(scaleMin, scaleMin, scaleMin) },
      tween.easing.quadIn,
      function()
        if not opponentGlowAlive(myGen) then return end
        pulse()
      end
      )
    end
    )
  end
  
  pulse()
end

function opponentGlowAlive(gen)
  return OpponentGlow
  and OpponentGlow.active
  and OpponentGlow.material
  and OpponentGlow._pulseGen == gen
end

function randomPleasantColor()
  local sets = {
    color(72,235,215),
    color(120,200,255),
    color(255,180,120),
    color(180,120,255),
    color(120,255,180),
    color(255,120,180),
    color(80,160,220),
    color(200,160,80)
  }
  return sets[math.random(1,#sets)]
end

SkyCycle = {
  t = 0,
  dur = 12,
  
  cur = {
    sky     = randomPleasantColor(),
    horizon = randomPleasantColor(),
    ground  = randomPleasantColor()
  },
  
  nxt = {
    sky = color(120,200,255),
    horizon = color(255,120,180),
    ground = color(80,160,220)
  }
}

function pickNextSkyColors()
  SkyCycle.nxt.sky     = randomPleasantColor()
  SkyCycle.nxt.horizon = randomPleasantColor()
  SkyCycle.nxt.ground  = randomPleasantColor()
end

function updateSkyCycle(dt)
  local s = SkyCycle
  s.t = s.t + dt
  
  if s.t > s.dur then
    s.t = 0
    
    -- lock in previous
    s.cur.sky     = s.nxt.sky
    s.cur.horizon = s.nxt.horizon
    s.cur.ground  = s.nxt.ground
    
    pickNextSkyColors()
  end
  
  local k = s.t / s.dur
  
  scene.sky.material.sky =
  lerpColor(s.cur.sky, s.nxt.sky, k)
  
  scene.sky.material.horizon =
  lerpColor(s.cur.horizon, s.nxt.horizon, k)
  
  scene.sky.material.ground =
  lerpColor(s.cur.ground, s.nxt.ground, k)
end

function lerpColor(a, b, t)
  return color(
  a.r + (b.r - a.r) * t,
  a.g + (b.g - a.g) * t,
  a.b + (b.b - a.b) * t,
  a.a + (b.a - a.a) * t
  )
end