--# BoardModel
HEX_DIRS_2D = {
  {0,-1},   -- north
  {-1,0},    -- northwest
  {-1,1},   -- southwest
  {0,1},    -- south
  {1,0},    -- southeast
  {1,-1},   -- northeast
}

BoardModel = class()

function BoardModel:init()
  self.hexCells = {}   -- key -> tile
  -- BoardModel
  self.lastCommit = nil
  self:newPendingCandidate()
end

function BoardModel:reset()
  -- hard clear all board state
  for k in pairs(self.hexCells) do
    self.hexCells[k] = nil
  end
  
  self.lastCommit = nil
  self._pendingCandidate = nil
  
  -- rebuild starting state
  self:spawnInitialTiles()
  self:newPendingCandidate()
end

function BoardModel:key(q,r)
  return q..","..r
end

function BoardModel:get(q,r)
  return self.hexCells[self:key(q,r)]
end

function BoardModel:set(q,r,tile)
  local key = self:key(q,r)
  self.hexCells[key] = {tile = tile,
    q = q,
    r = r
  }
end

function BoardModel:remove(q,r)
  self.hexCells[self:key(q,r)] = nil
end

function BoardModel:rotateTile(q, r, direction)
  local cell = self:get(q,r)
  if not cell then return end
  
  cell.tile:rotate(direction)
end

function BoardModel:allHexCellsAsFlatList()
  local list = {}
  for _,t in pairs(self.hexCells) do
    list[#list+1] = t
  end
  return list
end

function BoardModel:getCandidateCell()
  for _,cell in ipairs(self:allHexCellsAsFlatList()) do
    if cell.tile.kind == "candidate" then
      return cell
    end
  end
end

function BoardModel:spawnInitialTiles()
  self:set(0, 0,  TileModel("A", randomSideColors(), "normal"))
  self:set(1, 0,  TileModel("B", randomSideColors(), "normal"))
  self:set(0, 1,  TileModel("C", randomSideColors(), "normal"))
  self:set(-1, 0, TileModel("D", randomSideColors(), "normal"))
end

function BoardModel:playableHexes()
  local out = {}
  local seen = {}
  
  for _,cell in pairs(self.hexCells) do
    
    -- 👇 SKIP candidate tile
    if cell.tile.kind ~= "candidate" then
      
      for _,d in ipairs(HEX_DIRS_2D) do
        local q = cell.q + d[1]
        local r = cell.r + d[2]
        local k = self:key(q,r)
        
        if not self.hexCells[k] and not seen[k] then
          seen[k] = true
          out[#out+1] = {q=q, r=r}
        end
      end
      
    end
  end
  
  return out
end

function BoardModel:newPendingCandidate()
  self.pendingCandidate = TileModel("CAND", randomSideColors(), "candidate")
  self.pendingCandidate:rotate("left")
end

function BoardModel:moveCandidate(q,r)
  local candTile = nil
  
  -- remove old candidate but KEEP the tile object
  for k,cell in pairs(self.hexCells) do
    if cell.tile and cell.tile.kind == "candidate" then
      candTile = cell.tile
      self.hexCells[k] = nil
      break
    end
  end
  
  -- if none exists yet, create it once
  if not candTile then
    candTile = self.pendingCandidate or TileModel("CAND", randomSideColors(), "candidate")
    candTile.kind = "candidate"
    self.pendingCandidate = candTile
  end
  
  -- place SAME tile at new spot (preserves sides + rotation state)
  self:set(q,r,candTile)
  
  --self:testCandidateAgainstNeighbors()
end

function BoardModel:cloneOfScoringModel()
  -- returns a CLONE that is safe to rotate
  
  for _,cell in ipairs(self:allHexCellsAsFlatList()) do
    if cell.tile and cell.tile.kind == "candidate" then
      return cell.tile:clone(), cell.q, cell.r
    end
  end
  
  local pending = self:pendingCandidateModel()
  if pending then
    return pending:clone(), nil, nil
  end
  
  return nil, nil, nil
end

function BoardModel:tileMatchesAt(tileModel, q, r) 
  local count = 0
  
  for i,d in ipairs(HEX_DIRS_2D) do
    local nq = q + d[1]
    local nr = r + d[2]
    
    local neighbor = self:get(nq,nr)
    if neighbor and neighbor.tile then
      local otherTile = neighbor.tile
      local mySide = DIR_TO_SIDE[i]
      
      local oppDir = ((i+2)%6)+1
      local theirSide = DIR_TO_SIDE[oppDir]
      
      if tileModel.sides[mySide] ==
      otherTile.sides[theirSide] then
        count = count + 1
      end
    end
  end
  
  return count
end

function BoardModel:candidateMatches()
  local tile, q, r = self:cloneOfScoringModel()
  if not tile or not q or not r then return 0 end
  return self:tileMatchesAt(tile, q, r)
end

function BoardModel:debugCandidateContacts()
  local cand
  
  for _,cell in ipairs(self:allHexCellsAsFlatList()) do
    if cell.tile and cell.tile.kind == "candidate" then
      cand = cell
      break
    end
  end
  
  if not cand then
    print("NO CANDIDATE")
    return
  end
  
  print("=== DEBUG CONTACTS at", cand.q, cand.r, "===")
  
  for i,d in ipairs(HEX_DIRS_2D) do
    local nq = cand.q + d[1]
    local nr = cand.r + d[2]
    
    local neighbor = self:get(nq,nr)
    if neighbor and neighbor.tile then
      local myColor = cand.tile.sides[i]
      
      -- your *actual* facing side math
      local opp = ((i+2)%6)+1
      local theirColor = neighbor.tile.sides[opp]
      
      local ok = (myColor == theirColor)
      
      print(
      "DIR", i,
      "cand side", i, myColor,
      "| neigh at", nq, nr,
      "side", opp, theirColor,
      ok and "MATCH" or "NO"
      )
    end
  end
end

function BoardModel:testCandidateAgainstNeighbors()
  local cand
  
  for _,cell in ipairs(self:allHexCellsAsFlatList()) do
    if cell.tile.kind == "candidate" then
      cand = cell
      break
    end
  end
  
  if not cand then
    print("NO CANDIDATE")
    return
  end
  
  for i,d in ipairs(HEX_DIRS_2D) do
    local nq = cand.q + d[1]
    local nr = cand.r + d[2]
    
    local neigh = self:get(nq,nr)
    if neigh then
      print("---- comparing to neighbor at", nq, nr)
      runTileMatchTest(cand.tile, neigh.tile)
    end
  end
end

function runTileMatchTest(a, b)
  print("=== TILE MATCH TEST ===")
  print("A:", table.unpack(a.sides))
  print("B:", table.unpack(b.sides))
  
  local pass = true
  
  for i=1,6 do
    local opp = ((i+2)%6)+1
    local ok = a.sides[i] == b.sides[opp]
    
    print(
    "side", i,
    "vs", opp,
    ok and "PASS" or "FAIL"
    )
    
    if not ok then pass = false end
  end
  
  print(pass and "ALL PASS" or "SOME FAIL")
end

function BoardModel:commitScoringModel(tileModel, q, r)
  assert(tileModel, "commitScoringModel: no model")
  assert(q and r, "commitScoringModel: no position")

  local pointsScored = self:tileMatchesAt(tileModel, q, r)
  
  -- 1) clear existing candidate cell (if any)
  for k,cell in pairs(self.hexCells) do
    if cell.tile and cell.tile.kind == "candidate" then
      self.hexCells[k] = nil
      break
    end
  end
  
  -- 🔒 record where the move happened
  self.lastCommit = {
    q = q,
    r = r,
    points = pointsScored
  }
  
  -- 2) place a NORMAL tile (clone to freeze it)
  local placed = tileModel:clone()
  placed.kind = "normal"
  self:set(q, r, placed)
  
  -- 3) create next pending tile
  self:newPendingCandidate()
end

function BoardModel:pendingCandidateModel()
  return self.pendingCandidate
end

function BoardModel:serialize()
  local out = {
    tiles = {},
    pendingCandidate = nil
  }
  
  for _,cell in pairs(self.hexCells) do
    local tile = cell.tile
    if tile then
      out.tiles[#out.tiles+1] = {
        q = cell.q,
        r = cell.r,
        sides = copySides(tile.sides),
        kind = tile.kind
      }
    end
  end
  
  if self.pendingCandidate then
    out.pendingCandidate = {
      sides = copySides(self.pendingCandidate.sides)
    }
  end
  
  return out
end

function BoardModel:deserialize(data)
  assert(data and data.tiles, "BoardModel:deserialize bad data")
  
  -- hard reset, preserve object identity
  for k in pairs(self.hexCells) do
    self.hexCells[k] = nil
  end
  
  self.lastCommit = nil
  self.pendingCandidate = nil
  
  for _,t in ipairs(data.tiles) do
    local tile = TileModel(nil, copySides(t.sides), t.kind)
    self:set(t.q, t.r, tile)
  end
  
  if data.pendingCandidate then
    local c = TileModel("CAND", copySides(data.pendingCandidate.sides), "candidate")
    self.pendingCandidate = c
  else
    self:newPendingCandidate()
  end
end