
--# UI2D
UI2D = class()

function UI2D:init(board, renderer)
  self.board = board
  self.renderer = renderer
end

function UI2D:moveCandidate(q,r)
  self.board:moveCandidate(q,r)
end

function UI2D:playableHexTapped(touch)
  if touch.state ~= ENDED then return end
  
  local plays = self.board:playableHexes()
  
  for _,h in ipairs(plays) do
    local p = self.renderer:axialToPixel(h.q, h.r)
    
    local dx = touch.x - p.x
    local dy = touch.y - p.y
    
    if dx*dx + dy*dy < (self.renderer.size*0.8)^2 then
      return h.q, h.r
    end
  end
end

function UI2D:candidateMatches()
  return self.board:candidateMatches()
end

function UI2D:drawMatchCount()
  local cand
  
  for _,cell in ipairs(self.board:allHexCellsAsFlatList()) do
    if cell.tile and cell.tile.kind == "candidate" then
      cand = cell
      break
    end
  end
  
  if not cand then return end
  
  local p = self.renderer:axialToPixel(cand.q, cand.r)
  local n = self:candidateMatches()
  
  pushStyle()
  textAlign(CENTER)
  fontSize(36)
  fill(255)
  text(n, p.x, p.y)
  popStyle()
end

function UI2D:candidateTapped(touch)
  if touch.state ~= ENDED then return end
  print("tapped")
  for _,cell in ipairs(self.board:allHexCellsAsFlatList()) do
    if cell.tile and cell.tile.kind == "candidate" then
      local p = self.renderer:axialToPixel(cell.q, cell.r)
      
      local dx = touch.x - p.x
      local dy = touch.y - p.y
      
      if dx*dx + dy*dy < (self.renderer.size*0.8)^2 then
        return cell.q, cell.r
      end
      
      break
    end
  end
end

--# HexRenderer2D
HexRenderer2D = class()

function HexRenderer2D:init(board, size)
  self.board = board
  self.size = size or 40
  self.origin = vec2(WIDTH/2, HEIGHT/2)
  self._fillMesh = mesh()
  self._fillMesh:resize(18)
end

-- axial -> pixel (flat top)
function HexRenderer2D:axialToPixel(q,r)
  local s = self.size
  local x = s * (3/2 * q)
  local y = s * (math.sqrt(3) * (r + q/2))
  return self.origin + vec2(x,y)
end

function HexRenderer2D:update()
  -- draw playable hexes
  local plays = self.board:playableHexes()
  for _,h in ipairs(plays) do
    local p = self:axialToPixel(h.q, h.r)
    self:updatePlayable(p)
  end
  -- draw placed tiles
  for _,t in ipairs(self.board:allHexCellsAsFlatList()) do
    local p = self:axialToPixel(t.q, t.r)
    self:updateHexCell(p, t.tile)
  end
end

function HexRenderer2D:updateHexCell(hexCenter, tile)
  local r = self.size
  local pts = {}
  
  for i=0,5 do
    local a = math.rad(-60*i + 120)
    pts[i+1] = hexCenter + vec2(
    r*math.cos(a),
    r*math.sin(a)
    )
  end
  
  local m = self._fillMesh
  
  local k=1
  for i=1,6 do
    local a = pts[i]
    local b = pts[i%6+1]
    
    m:vertex(k,hexCenter); m:color(k,200,200,255); k=k+1
    m:vertex(k,a);      m:color(k,200,200,255); k=k+1
    m:vertex(k,b);      m:color(k,200,200,255); k=k+1
  end
  
  noStroke()
  m:draw()
  
  strokeWidth(self.size * 0.25)  -- thick, obvious
  
  local inset = (self.size * 0.25) * 0.7  -- half stroke
  
  for i=1,6 do
    local a = pts[i]
    local b = pts[i%6+1]
    
    -- move edge inward
    local mid = (a + b) * 0.5
    local dir = (hexCenter - mid)
    dir = dir:normalize() * inset
    
    -- inward shift
    local a2 = a + dir
    local b2 = b + dir
    
    -- shorten ends
    local edgeDir = (b2 - a2)
    edgeDir = edgeDir:normalize()
    
    local trim = inset + 2 -- same amount feels right visually
    
    a2 = a2 + edgeDir * trim
    b2 = b2 - edgeDir * trim
    
    stroke(0)
    strokeWidth(2)
    
    if tile then
      local sideName = tile.sides[i]
      stroke(EDGE_COLORS[sideName])
      strokeWidth(self.size * 0.35)
    end
    
    line(a2.x, a2.y, b2.x, b2.y)
  end
end

function HexRenderer2D:updatePlayable(center)
  local r = self.size
  local pts = {}
  
  for i=0,5 do
    local a = math.rad(-60*i + 120)
    pts[i+1] = center + vec2(
    r*math.cos(a),
    r*math.sin(a)
    )
  end
  
  stroke(255,200,80)
  strokeWidth(3)
  noFill()
  
  for i=1,6 do
    local a = pts[i]
    local b = pts[i%6+1]
    line(a.x,a.y,b.x,b.y)
  end
end

function runTileMatchTest(a, b)
  print("=== TILE MATCH TEST ===")
  print("A:", table.unpack(a.sides))
  print("B:", table.unpack(b.sides))
  
  local pass = true
  
  for i=1,6 do
    local opp = ((i+2)%6)+1
    local ok = a.sides[i] == b.sides[opp]
    
    print(
    "side", i,
    "vs", opp,
    ok and "PASS" or "FAIL"
    )
    
    if not ok then pass = false end
  end
  
  print(pass and "ALL PASS" or "SOME FAIL")
end

--[[
--# Main
require(asset.documents.FullOrbitViewer)
require(asset.documents.immediateUI)

function setup()
  viewer.mode = FULLSCREEN
  
  board = BoardModel()
  board:spawnInitialTiles()
  
  renderer = HexRenderer2D(board, 42)
  ui = UI2D(board, renderer)
end

function draw()
  background(20,20,40)
  renderer:update()
  ui:drawMatchCount()
end

function touched(t)
  if t.state ~= ENDED then return end
  
  -- rotate candidate if tapped
  local q,r = ui:candidateTapped(t)
  if q then
    board:rotateTile(q, r, "left")
    return
  end
  
  -- otherwise move candidate
  local q2,r2 = ui:playableHexTapped(t)
  if q2 then
    ui:moveCandidate(q2,r2)
  end
end]]