---------------------------------------------------------
-- 2) IMMEDIATE-MODE BUTTON SYSTEM (with pressing/swapping)
---------------------------------------------------------
---------------------------
-- 2.1) Sensor class created by JMV38
---------------------------
Sensor = class()
local abs = math.abs

function Sensor:init(t)
    self.enabled = true
    self.extra   = t.extra or 0
    self.touches = {}
    self:setParent(t)
    self.events  = {}
    self.doNotInterceptTouches = false
end

function Sensor:setParent(t)
    local p = t.parent or self.parent
    if p.x and p.y and p.w and p.h then
        self.parent = p
    else
        error("Sensor parent must have x,y,w,h coordinates")
    end
    
    self.xywhMode = t.xywhMode or CORNER
    if self.xywhMode == CENTER then
        self.xywh = self.xywhCENTER
    elseif self.xywhMode == CORNER then
        self.xywh = self.xywhCORNER
    elseif self.xywhMode == RADIUS then
        self.xywh = self.xywhRADIUS
    end
end

function Sensor:register(eventName, update, callback)
    table.insert(self.events, {
        name = eventName, 
        callback = callback,
        update   = update
    })
end

-- onTouch: “at least one finger is inside” => state=true
function Sensor:onTouch(callback)
    self:register("onTouch", self.touchUpdate, callback)
end

function Sensor.touchUpdate(event, sensor, t)
    sensor.touching = sensor.touching or {}
    
    -- Track if a finger is inside
    if sensor:inbox(t) and (t.state == BEGAN or t.state == MOVING) then
        sensor.touching[t.id] = true
    elseif (t.state == ENDED or t.state == CANCELLED) then
        sensor.touching[t.id] = nil
    end
    
    -- Check overall “any finger inside?” state
    local newState = false
    for _, inside in pairs(sensor.touching) do
        if inside then
            newState = true
            break
        end
    end
    
    -- If changed, fire callback
    if newState ~= event.state then
        event.state = newState
        event.touch = t
        event:callback()
    end
end

-- drag gesture
function Sensor:onDrag(callback)
    self:register("onDrag", self.dragUpdate, callback)
end
function Sensor.dragUpdate(event,self,t)
    if self.touches[t.id] then
        event.touch = t
        event:callback()
    end
end

-- long press gesture
function Sensor:onLongPress(callback)
    self:register("onLongPress", self.longPressUpdate, callback)
end

function Sensor.longPressUpdate(event, self, t)
    local tmin = 1   -- seconds required to count as long press
    if self.touches[t.id] then -- the touch must have started on this sensor
        if t.state == BEGAN then
            event.totalMove = 0
            event.cancel = false
            event.id = t.id
            event.tween = tween.delay(tmin, function()
                event.tween = nil
                if event.totalMove > 10 or event.id ~= t.id then
                    event.cancel = true
                end
                if event.cancel then return end
                event:callback()
            end)
        elseif t.state == MOVING and event.id == t.id then
            -- integrate finger movement
            event.totalMove = (event.totalMove or 0)
            + abs(t.deltaX) + abs(t.deltaY)
        elseif (t.state == ENDED or t.state == CANCELLED) and event.id == t.id then
            event.cancel = true
            if event.tween then tween.stop(event.tween) end
        end
    end
end

-- onTap: short press with minimal movement
function Sensor:onTap(callback)
    self:register("onTap", self.tapUpdate, callback)
end

function Sensor.tapUpdate(event, sensor, t)
    if t.state == BEGAN and sensor:inbox(t) then
        sensor.touches[t.id] = true
        event.totalMove = 0
        event.t0 = ElapsedTime
        
    elseif t.state == MOVING and sensor.touches[t.id] then
        event.totalMove = (event.totalMove or 0) + abs(t.deltaX) + abs(t.deltaY)
        
    elseif t.state == ENDED and sensor.touches[t.id] then
        sensor.touches[t.id] = nil
        local duration = ElapsedTime - (event.t0 or 0)
        local movement = event.totalMove or 9999
        
        if duration < 0.5 and movement < 20 then
            event.touch = t
            event:callback()
        end
    end
end

function Sensor:touched(t)
    if not self.enabled then return end
    
    if t.state == BEGAN and self:inbox(t) then
        self.touches[t.id] = true
    end
    
    for _, event in ipairs(self.events) do
        event:update(self, t)
    end
    
    local intercepted = self.touches[t.id]
    if self.doNotInterceptTouches then
        intercepted = false
    end
    
    if t.state == ENDED or t.state == CANCELLED then
        self.touches[t.id] = nil
    end
    return intercepted
end

function Sensor:xywhCORNER()
    local p = self.parent
    return p.x, p.y, p.w, p.h
end

function Sensor:xywhCENTER()
    local p = self.parent
    return p.x - p.w/2, p.y - p.h/2, p.w, p.h
end

function Sensor:xywhRADIUS()
    local p = self.parent
    return p.x - p.w, p.y - p.h, p.w*2, p.h*2
end

function Sensor:inbox(t)
    local x,y,w,h = self:xywh()
    return (t.x >= x - self.extra and t.x <= x + w + self.extra
    and t.y >= y - self.extra and t.y <= y + h + self.extra)
end
    
Sensor._touchDB = Sensor._touchDB or {
    touches = {},   -- [id] = record
    seq = 0
}

local function _now()
    return ElapsedTime
end

function Sensor.captureTouch(t)
    -- t is Codea touch or your _IMUI.localTouch; both have id,state,x,y,deltaX,deltaY
    local db = Sensor._touchDB
    db.seq = db.seq + 1
    
    local id = t.id
    local r = db.touches[id]
    
    if t.state == BEGAN then
        r = {
            id = id,
            beganX = t.x, beganY = t.y,
            beganTime = _now(),
            moved = 0,
            lastState = BEGAN,
            x = t.x, y = t.y,
            dx = 0, dy = 0,
            endedX = nil, endedY = nil,
            endedTime = nil,
            seqBegan = db.seq,
            seqEnded = nil
        }
        db.touches[id] = r
        return
    end
    
    if not r then
        -- defensive: if we ever see MOVING/ENDED without BEGAN, still track safely
        r = {
            id = id,
            beganX = t.x, beganY = t.y,
            beganTime = _now(),
            moved = 0,
            lastState = t.state,
            x = t.x, y = t.y,
            dx = t.deltaX or 0, dy = t.deltaY or 0,
            endedX = nil, endedY = nil,
            endedTime = nil,
            seqBegan = nil,
            seqEnded = nil
        }
        db.touches[id] = r
    end
    
    r.dx = t.deltaX or 0
    r.dy = t.deltaY or 0
    r.x  = t.x
    r.y  = t.y
    r.lastState = t.state
    
    if t.state == MOVING then
        r.moved = (r.moved or 0) + math.abs(r.dx) + math.abs(r.dy)
    elseif t.state == ENDED or t.state == CANCELLED then
        r.endedX = t.x
        r.endedY = t.y
        r.endedTime = _now()
        r.seqEnded = db.seq
        -- NOTE: do NOT delete here; leave it for inspection
    end
end

function Sensor.getTouchDB()
    return Sensor._touchDB
end