


---------------------------
-- 2.2) Immediate-Mode UI
---------------------------
-- Declare the global toggle. (Change the default as needed.)
parameter.boolean("showCoords", false, function(v) _IMUI.showCoords = v end)

-- We'll keep all internal state in _IMUI
_IMUI = {
    sensor          = nil,  -- We'll create on-demand
    pressedLocation = nil,  -- Where user is currently holding
    lastTapLocation = nil,   -- Where user just tapped
    showCoords = showCoords,
    LEFT_ROUNDED  = 3,
    RIGHT_ROUNDED = 12,
    NO_ROUNDING   = 0,
    popupState = nil,
    popupContainer = nil,
    sliderState = {},      -- stores current values,,,
    sliderLastFrame = {},    -- helps detect first-time init per frame
    sliderSensors = {},      -- one sensor per slider
    bounds ={},
    activeColorWheelTouches = {},
    _btnSpecs = {}
}

function immediateUI(touch)
    Sensor.captureTouch(touch)

    -- Record an initial touch‐down location.
    if touch.state == BEGAN then
        _IMUI.lastTouchDownLocation = vec2(touch.x, touch.y)
    end
    -- Record a touch‐up location.
    if touch.state == ENDED then
        _IMUI.lastTouchUpLocation = vec2(touch.x, touch.y)
        -- and lastTapLocation for onTap logic
        _IMUI.lastTapLocation = vec2(touch.x, touch.y)
    end
    
    -- Translate into localTouch (no offsets needed)
    _IMUI.localTouch = {
        x      = touch.x,
        y      = touch.y,
        id     = touch.id,
        state  = touch.state,
        deltaX = touch.deltaX,
        deltaY = touch.deltaY
    }
end

-- Convert a global tap to local coords if we're inside a uiPanel()
local function _imuiLocalTap()
    local tap = _IMUI.lastTapLocation
    if not tap then return nil end
    local spec = _IMUI.currentContainerSpecs
    if not spec then return tap end
    return vec2(tap.x - spec.offset.x, tap.y - spec.offset.y)
end

-- Draw a 10-pixel grid starting from the lower left.
function drawGrid()
    pushStyle()
    strokeWidth(1)
    for i = 0, WIDTH, 10 do
        if math.floor(i/10) % 2 == 0 then
            stroke(255, 100, 100, 220)
        else
            stroke(255, 100, 100, 100)
        end
        line(i, 0, i, HEIGHT)
    end
    
    for j = 0, HEIGHT, 10 do
        if math.floor(j/10) % 2 == 0 then
            stroke(255, 100, 100, 220)
        else
            stroke(255, 100, 100, 100)
        end
        line(0, j, WIDTH, j)
    end
    if _IMUI and _IMUI.buttonSensors then
        stroke(0, 255, 0, 200)
        noFill()
        strokeWidth(3)
        local prevMode = rectMode()
        rectMode(CORNER)
        for key, sensor in pairs(_IMUI.buttonSensors) do
            if sensor.parent then
                local x, y, w, h = sensor:xywh()
                rect(x, y, w, h)
            end
        end
        rectMode(prevMode)
    end
    popStyle()
end


_IMUI.dispatchTouchTo = function(sensor)
    if not _IMUI.localTouch then return end
    local raw = _IMUI.localTouch
    local spec = _IMUI.currentContainerSpecs
    
    -- 1) If we have a panel spec, gate by its global bounds:
    if spec then
        local ox, oy = spec.offset.x, spec.offset.y
        if raw.x < ox or raw.x > ox + spec.w
        or raw.y < oy or raw.y > oy + spec.h then
            return
        end
    end
    
    -- 2) Offset into local coords (if spec exists)
    local t = raw
    if spec then
        t = {
            x      = raw.x - spec.offset.x,
            y      = raw.y - spec.offset.y,
            id     = raw.id,
            state  = raw.state,
            deltaX = raw.deltaX,
            deltaY = raw.deltaY
        }
    end
    
    -- 3) Deliver to the sensor
    sensor:touched(t)
end

-- Fits `textToFit` into a w×h box centered at (x, y) by adjusting fontSize
textFitToRect = function(textToFit, x, y, w, h)
    -- 1) stash style, set up wrapping and centering
    pushStyle()
    rectMode(CENTER)
    textWrapWidth(w)
    
    -- 2) binary-search between minSize and maxSize
    local minSize, maxSize = 1, 1000
    local mid, tw, th
    
    while maxSize - minSize > 0.5 do
        mid = (minSize + maxSize) * 0.5
        fontSize(mid)
        tw, th = textSize(textToFit)
        if th > h then
            -- too tall → shrink upper bound
            maxSize = mid
        else
            -- fits (or is too short) → raise lower bound
            minSize = mid
        end
    end
    
    -- 3) use the largest size that still fits
    fontSize(minSize)
    text(textToFit, x, y)
    
    -- 4) restore previous style (including the old font size)
    popStyle()
end

-- Compute default button dimensions from text (with a default margin).
_IMUI.getButtonDimensions = function(label)
    local margin = 45
    local tw, th = textSize(label)
    return tw + margin, th + margin * 0.5
end

-- Reset grid flags; call this at the start of button() so that the grid is drawn once per frame.
_IMUI.resetGridFlag = function()
    local currentFrame = math.floor(ElapsedTime * 1000)
    if _IMUI.lastGridFrame ~= currentFrame then
        _IMUI.gridDrawn = false
        _IMUI.lastGridFrame = currentFrame
    end
    if _IMUI.showCoords and not _IMUI.gridDrawn then
        drawGrid()
        _IMUI.gridDrawn = true
    end
end

-- Get the text position based on the current rectMode.
_IMUI.getButtonTextPosition = function(x, y, w, h)
    local mode = rectMode()
    if mode == CORNER then
        return x + w/2, y + h/2
    elseif mode == CENTER or mode == RADIUS then
        return x, y
    else
        return x + w/2, y + h/2
    end
end

-- Draw the button background using the roundedRectangle function.
_IMUI.drawButtonBackground = function(x, y, w, h, crOptions)
    roundedRectangle{
        x = x,
        y = y,
        w = w,
        h = h,
        corners = crOptions.corners,
        radius = crOptions.radius
    }
end

-- Check tap and release locations to fire a button callback.
_IMUI.handleButtonTouch = function(x, y, w, h, callback)
    local tapLoc = _IMUI.lastTapLocation
    if tapLoc and pointInRectByCurrentMode(tapLoc.x, tapLoc.y, x, y, w, h) then
        if callback then callback() end
        _IMUI.lastTapLocation = nil
        _IMUI.lastTouchUpLocation = nil
        _IMUI.lastTouchDownLocation = nil
        return true
    elseif _IMUI.lastTouchDownLocation and _IMUI.lastTouchUpLocation and
    pointInRectByCurrentMode(_IMUI.lastTouchDownLocation.x, _IMUI.lastTouchDownLocation.y, x, y, w, h) and
    pointInRectByCurrentMode(_IMUI.lastTouchUpLocation.x, _IMUI.lastTouchUpLocation.y, x, y, w, h) then
        if callback then callback() end
        _IMUI.lastTouchUpLocation = nil
        _IMUI.lastTouchDownLocation = nil
        return true
    end
    return false
end

-- Select the button colors based on whether it’s stickyState, pressed, or active.
-- Select colors for normal / pressed / active (sticky) states.
-- Accepts optional crTable.highlightFG / crTable.highlightBG to override the
-- default “swap fill↔stroke” highlight look.
_IMUI.getButtonColors = function(isSticky, isPressed, crTable, originalFill, originalStroke)
    local hiBG = crTable and crTable.highlightBG  -- BG while pressed/active
    local hiFG = crTable and crTable.highlightFG  -- text while pressed/active
    
    -- sensible defaults if not provided: invert the palette
    hiBG = hiBG or originalStroke
    hiFG = hiFG or originalFill
    
    local btnFill, btnStroke, textColor
    
    if isSticky then
        -- sticky buttons show highlight when active OR while currently pressed
        local activeLike = (crTable and crTable.active) or false
        if isPressed or activeLike then
            btnFill   = hiBG
            btnStroke = originalStroke
            textColor = hiFG
        else
            btnFill   = originalFill
            btnStroke = originalStroke
            textColor = originalStroke
        end
    else
        -- non-sticky: only while pressed
        if isPressed then
            btnFill   = hiBG
            btnStroke = originalStroke
            textColor = hiFG
        else
            btnFill   = originalFill
            btnStroke = originalStroke
            textColor = originalStroke
        end
    end
    return btnFill, btnStroke, textColor
end

function _IMUI._drawStoredPopup()
    local p = _IMUI.pendingPopup
    if not p then return end
    
    local w, h = _IMUI.getButtonDimensions(p.text)
    local cx, cy = WIDTH / 2, HEIGHT / 2
    local x, y = cx - w/2, cy - h/2
    
    pushStyle()
    fill(0, 180)
    noStroke()
    rectMode(CORNER)
    rect(0, 0, WIDTH, HEIGHT) -- backdrop
    rectMode(CENTER)
    button(p.text, cx, cy)    -- popup content (text only for now)
    popStyle()
    
    -- Store hitbox for dismissal
    p._x, p._y, p._w, p._h = x, y, w, h
    
    -- Mark that it has been drawn so we allow dismissal
    _IMUI.pendingPopupDrawn = true
end

-- End helper functions

-------------------------------------------------------------------
-- The button() function that owns a 2-mesh spec per button
function button(label, x, y, w, h, callback, cornerRadius)
    -- unique key: explicit key in options or fallback to file:line:label
    local widgetKey
    if type(cornerRadius) == "table" and cornerRadius.key then
        widgetKey = cornerRadius.key
    else
        local info = debug.getinfo(2, "Sl")
        widgetKey = info.short_src .. ":" .. info.currentline .. ":" .. label
    end
    
    pushStyle()
    rectMode(CENTER)
    
    -- Missing dims → compute from text
    if not w or type(w) == "function" then
        callback     = w
        cornerRadius = h
        w, h = _IMUI.getButtonDimensions(label)
    end
    
    _IMUI.resetGridFlag()
    
    -- sensors (unchanged)
    _IMUI.buttonSensors = _IMUI.buttonSensors or {}
    _IMUI.buttonPressed = _IMUI.buttonPressed or {}
    
    local sensor = _IMUI.buttonSensors[widgetKey]
    if not sensor then
        sensor = Sensor{
            parent   = { x=x, y=y, w=w, h=h },
            xywhMode = CENTER
        }
        sensor:onTouch(function(event)
            local isSticky  = type(cornerRadius)=="table" and cornerRadius.stickyState
            local isPressed = _IMUI.buttonPressed[widgetKey] == true
            if isSticky then
                -- sticky tracks “active” in statefulButton; here we only handle press visuals
                if isPressed then _IMUI.buttonPressed[widgetKey] = true
                else              _IMUI.buttonPressed[widgetKey] = false end
                return
            end
            if event.state and event.touch.state == BEGAN then
                _IMUI.buttonPressed[widgetKey] = true
            elseif not event.state and (event.touch.state == ENDED or event.touch.state == CANCELLED) then
                _IMUI.buttonPressed[widgetKey] = false
            end
        end)
        sensor:onTap(function() if callback then callback() end end)
        _IMUI.buttonSensors[widgetKey] = sensor
    end
    
    -- keep sensor in sync with current rect
    sensor.parent.x, sensor.parent.y, sensor.parent.w, sensor.parent.h = x, y, w, h
    _IMUI.dispatchTouchTo(sensor)
    
    local isPressed = _IMUI.buttonPressed[widgetKey] == true
    
    -- quick-reuse cache per button key
    local crTable
    if type(cornerRadius) == "table" then
        crTable = cornerRadius
    else
        crTable = { radius = cornerRadius or 18, corners = 15 }
    end
    local isSticky = (crTable.stickyState == true)
    -- quick-reuse cache per button key
    _IMUI._btnCache = _IMUI._btnCache or {}
    local cache = _IMUI._btnCache[widgetKey]
    
    if cache
    and cache.w == w and cache.h == h
    and cache.label == label
    and cache.fill == color(fill())
    and cache.stroke == color(stroke())
    and cache.strokeW == strokeWidth()
    then
        -- choose mesh based on CURRENT state, not the previously cached mesh
        local spec = _IMUI._btnSpecs[widgetKey]
        if spec then
            local drawActive = (isSticky and (crTable.active == true)) or isPressed
            local _, _, txtCol = _IMUI.getButtonColors(isSticky, isPressed, crTable,
            color(fill()), color(stroke()))
            local m = drawActive and spec.activeMesh or spec.normalMesh
            
            pushMatrix(); translate(x, y); m:draw(); popMatrix()
            
            local textX, textY = _IMUI.getButtonTextPosition(x, y, w, h)
            fill(txtCol); textMode(CENTER); textAlign(CENTER)
            text(label, textX, textY)
            popStyle()
            return widgetKey
        end
    end
    
    
    -- capture the caller’s current palette (these drive rebuilds)
    local currentFill     = color(fill())
    local currentStroke   = color(stroke())
    local currentStrokeW  = strokeWidth()
    
    if _IMUI.showCoords then currentFill = color(170,170) end
    
    -- normalize options
    local crTable
    if type(cornerRadius) == "table" then
        crTable = cornerRadius
    else
        crTable = { radius = cornerRadius or 18, corners = 15 }
    end
    local isSticky = (crTable.stickyState == true)
    
    -- resolve normal/active colors (no drawing yet)
    local btnFill, btnStroke, textColor =
    _IMUI.getButtonColors(isSticky, isPressed, crTable, currentFill, currentStroke)
    
    -- build/reuse the two-mesh spec for this button + style
    -- Highlighted (active/pressed) wants swapped palette (hiBG/hiFG); we already computed that in getButtonColors().
    -- For the geometry spec we only care about button fills (normal vs active) and same stroke color.
    -- Use btnFill for "active" and currentFill for "normal" to force a distinct mesh pair.
    local radius  = crTable.radius or 18
    local corners = crTable.corners or 15
    
    -- Normal colors come from current fill/stroke
    local normalFill   = currentFill
    local normalStroke = currentStroke
    
    -- Active/pressed colors:
    -- prefer explicit highlightBG/FG if provided, else swap
    local hiBG = crTable.highlightBG
    local hiFG = crTable.highlightFG
    
    -- use getButtonColors to determine what the pressed state would look like
    local _, pressedStroke, pressedText =
    _IMUI.getButtonColors(isSticky, true, crTable, currentFill, currentStroke)
    
    -- now decide the pressed fill/stroke for the mesh:
    --  • if highlightBG/FG exist → use them
    --  • else → swap fill/stroke, except if getButtonColors
    --    already tells us to keep the stroke unified (segmented control)
    local activeFill   = hiBG or pressedStroke or normalStroke
    local activeStroke = hiFG or currentStroke  -- keep stroke unified if control needs it
    
    local spec = _getButtonSpecs(
    widgetKey,
    w, h, radius, corners,
    normalFill, normalStroke, currentStrokeW,
    activeFill, activeStroke
    )
    
    -- choose which mesh to draw
    local drawActive = isSticky and (crTable.active == true) or isPressed
    local m = drawActive and spec.activeMesh or spec.normalMesh
    
    -- draw background (mesh) in place
    pushMatrix()
    translate(x, y)
    m:draw()
    popMatrix()
    
    -- draw label (text color chosen earlier)
    local textX, textY = _IMUI.getButtonTextPosition(x, y, w, h)
    fill(textColor)
    textMode(CENTER); textAlign(CENTER)
    text(label, textX, textY)
    _IMUI._btnCache[widgetKey] = {
        w=w, h=h,
        label=label,
        fill=color(fill()),
        stroke=color(stroke()),
        strokeW=strokeWidth()
        -- no mesh here; we select from spec each frame
    }
    popStyle()
    
    -- optional coords overlay
    if _IMUI.showCoords then
        pushStyle()
        noStroke()
        font("AmericanTypewriter-Bold"); fontSize(16)
        fill(60,170); rect(textX, textY, 90, 20)
        fill(60);     text(string.format("(%d, %d)", x//1, y//1), textX+1, textY-1)
        fill(240);    text(string.format("(%d, %d)", x//1, y//1), textX,   textY)
        popStyle()
    end
    
    return widgetKey
end

-- Wrapper around IMUI.button with a boolean state.
-- When `state` is true, it swaps fill() and stroke() before drawing,
-- then restores them. Otherwise it just calls button as-is.
function statefulButton(label, state, x, y, w, h, callback, cornerRadiusOrOptsTable)
    if type(cornerRadiusOrOptsTable) == "table" then
        cornerRadiusOrOptsTable.stickyState = true
        cornerRadiusOrOptsTable.active = state
    else
        cornerRadiusOrOptsTable = {
            radius = cornerRadiusOrOptsTable,
            stickyState = true,
            active = state
        }
    end
    button(label, x, y, w, h, callback, cornerRadiusOrOptsTable)
end

function uiPanel(cx, cy, w, h, drawFn)
    -- draw the panel background (same as before)
    pushStyle()
    rectMode(CENTER)
    roundedRectangle{ x=cx, y=cy, w=w, h=h, corners=15, radius=12 }
    popStyle()
    
    -- store the current container spec (offset + size)
    _IMUI.currentContainerSpecs = {
        offset = vec2(cx - w/2, cy - h/2),
        w      = w,
        h      = h
    }
    
    -- draw children in local space
    pushMatrix()
    translate(_IMUI.currentContainerSpecs.offset.x,
    _IMUI.currentContainerSpecs.offset.y)
    drawFn()
    popMatrix()
    
    -- clear
    _IMUI.currentContainerSpecs = nil
end

-- wrapper for text(...) so you don't have to manually reverse fill and stroke if you want to draw text the same color as button text
function imuiText(aString, x, y)
    pushStyle()
    fill(stroke())
    text(aString, x, y)
    popStyle()
end

-- segmentedControl creates buttons side by side where only one can be active
-- The function call is:
-- segmentedControl(x, y, totalWidth, height, "Label1", action1, "Label2", action2, ...)
-- segmentedControl creates buttons side by side where only one can be active
-- Call forms supported:
--   segmentedControl(x, y, w, h, opts, "Label1", action1, "Label2", action2, ...)
--   segmentedControl(x, y, w, h,      "Label1", action1, "Label2", action2, ...)
--   segmentedControl(x, y, opts,      "Label1", action1, "Label2", action2, ...)  -- auto size
function segmentedControl(x, y, totalWidth, h, optsOrLabel, ...)
    local opts
    local args       -- { label1, cb1, label2, cb2, ... }
    local wNum, hNum -- numeric width/height (may be nil -> auto)
    
    -- Case 1: third argument is the options table, no explicit w/h
    if type(totalWidth) == "table" then
        opts  = totalWidth
        wNum  = nil
        hNum  = nil
        args  = { h, optsOrLabel, ... }  -- h is first label
        -- Case 2: fourth argument is the options table (have width, no explicit height)
    elseif type(h) == "table" then
        opts  = h
        wNum  = totalWidth
        hNum  = nil
        args  = { optsOrLabel, ... }
        -- Case 3: fifth argument is the options table (have width + height)
    elseif type(optsOrLabel) == "table" then
        opts  = optsOrLabel
        wNum  = totalWidth
        hNum  = h
        args  = { ... }
        -- Original simple form: w,h then labels/callbacks
    else
        opts  = nil
        wNum  = totalWidth
        hNum  = h
        args  = { optsOrLabel, ... }
    end
    
    local numSegments = #args / 2
    if numSegments < 1 then return end
    
    -- If dimensions aren't provided or not numeric, compute them from labels.
    if type(wNum) ~= "number" or type(hNum) ~= "number" then
        local maxTextWidth  = 0
        local maxTextHeight = 0
        for i = 1, numSegments do
            local label = args[(i-1)*2 + 1]
            local tw, th = textSize(label)
            if tw > maxTextWidth  then maxTextWidth  = tw end
            if th > maxTextHeight then maxTextHeight = th end
        end
        local padding = 45  -- default margin
        wNum = (maxTextWidth + padding) * numSegments
        hNum = maxTextHeight + padding * 0.5
    end
    
    local totalWidthNum = wNum
    local hNumFinal     = hNum
    
    -- Create a unique ID for this segmented control and initialize its selected index if needed.
    _IMUI.segmentedControls = _IMUI.segmentedControls or {}
    local id = "segmentedControl:" .. x .. ":" .. y .. ":" .. tostring(totalWidthNum) .. ":" .. tostring(hNumFinal)
    if _IMUI.segmentedControls[id] == nil then
        _IMUI.segmentedControls[id] = 1  -- default to the first segment
    end
    local selectedSegment = _IMUI.segmentedControls[id]
    
    local segmentWidth = totalWidthNum / numSegments
    local mode = rectMode()
    
    for i = 1, numSegments do
        local label    = args[(i-1)*2 + 1]
        local callback = args[(i-1)*2 + 2]
        
        local segX
        if mode == CORNER then
            -- In CORNER mode, x is the bottom-left corner.
            segX = x + (i - 1) * segmentWidth + segmentWidth/2
        else
            -- In CENTER or RADIUS mode, x is the overall center.
            segX = x - totalWidthNum/2 + (i - 0.5) * segmentWidth
        end
        
        -- Determine the desired corner rounding:
        local corners
        if i == 1 then
            corners = _IMUI.LEFT_ROUNDED        -- Only left-side corners rounded
        elseif i == numSegments then
            corners = _IMUI.RIGHT_ROUNDED       -- Only right-side corners rounded
        else
            corners = _IMUI.NO_ROUNDING         -- Middle buttons are square
        end
        
        local segmentTable = {
            corners     = corners,
            radius      = 18,
            stickyState = true,
            key         = id .. ":" .. i
        }
        
        -- Inject shared highlight colors if provided
        if opts then
            if opts.highlightBG then segmentTable.highlightBG = opts.highlightBG end
            if opts.highlightFG then segmentTable.highlightFG = opts.highlightFG end
        end
        
        if i == selectedSegment then
            segmentTable.active = true
        end
        
        button(label, segX, y, segmentWidth, hNumFinal, function()
            _IMUI.segmentedControls[id] = i  -- update the selected segment
            segmentTable.active = true
            if callback then callback() end
        end, segmentTable)
    end
end

-- showPopup(message, onDismiss)
-- Draws a centered popup with the given text message.
-- Remains visible until user taps outside the box.
-- When dismissed:
--   - Calls `onDismiss()` (if provided)
--   - Returns a state object: { dismissed = true/false, active = true/false }

-- showPopup(message, onDismiss)
-- Draws a modal popup with text.
-- Dismisses on tap outside the popup.
-- Returns: { dismissed = true } on the frame it was dismissed

function showPopup(message, onDismiss)
    -- Create popup if not present
    if not _IMUI.popupState then
        _IMUI.popupState = {
            text = message,
            onDismiss = onDismiss,
            dismissed = false
        }
    end
    
    local s = _IMUI.popupState
    
    -- Draw popup
    local w, h = _IMUI.getButtonDimensions(s.text)
    local cx, cy = WIDTH / 2, HEIGHT / 2
    local x, y = cx - w/2, cy - h/2
    
    pushStyle()
    rectMode(CENTER)
    button(s.text, cx, cy)
    popStyle()
    
    -- Store position for touch hitbox
    s._x, s._y, s._w, s._h = x, y, w, h
    
    -- Return state object
    return _IMUI.popupState
end

function getSlider(name)
    return _IMUI.sliderState[name]
end

function setSlider(name, value)
    if _IMUI.sliderState[name] ~= nil then
        _IMUI.sliderState[name] = value
    end
end

function slider(name, x, y, width, min, max, initial, callback, textColorOverride)
    assert(name, "slider(...) error: name is required")
    
    local function clamp(val, minVal, maxVal)
        if val == nil then val = minVal end
        return math.max(minVal, math.min(maxVal, val))
    end
    
    -- Detect start of new frame
    local t = ElapsedTime
    if t ~= _IMUI._lastDrawTime then
        _IMUI._lastDrawTime = t
        _IMUI.drawCallCount = 0
        _IMUI.sliderLastFrame = {}
    end
    _IMUI.drawCallCount = _IMUI.drawCallCount + 1
    
    -- State init
    if _IMUI.sliderState[name] == nil then
        _IMUI.sliderState[name] = clamp(initial, min, max)
    end
    local value = _IMUI.sliderState[name]
    
    -- Layout
    local sliderLength = width or 200
    local knobRadius = 12
    local leftX = x - sliderLength/2
    local rightX = x + sliderLength/2
    local textY = y + 24
    
    local function posFromValue(v)
        v = (v == nil) and min or v
        return leftX + ((v - min) / (max - min)) * sliderLength
    end
    
    local function valueFromPos(px)
        return clamp(((px - leftX) / sliderLength) * (max - min) + min, min, max)
    end
    
    -- Sensor setup
    local sensor = _IMUI.sliderSensors[name]
    if not sensor then
        sensor = Sensor{
            parent = { x = posFromValue(value), y = y, w = knobRadius*2, h = knobRadius*2 },
            xywhMode = CENTER
        }
        
        sensor:onDrag(function(event)
            local t = event.touch
            -- map the finger’s X directly into [leftX, rightX]
            local px = clamp(t.x, leftX, rightX)
            local newVal = valueFromPos(px)
            -- move knob instantly under the finger
            sensor.parent.x = posFromValue(newVal)
            if newVal ~= _IMUI.sliderState[name] then
                _IMUI.sliderState[name] = newVal
                if callback then callback(newVal) end
            end
        end)
        
        _IMUI.sliderSensors[name] = sensor
    end
    
    -- Update position
    --tsensor.parent._inContainer = _IMUI.inContainer == true
    sensor.parent.x = posFromValue(_IMUI.sliderState[name])
    sensor.parent.y = y
    _IMUI.dispatchTouchTo(sensor)
    
    -- Draw the track
    pushStyle()
    if strokeWidth() == 0 then
        print("warning: slider '"..name.."' is being drawn with strokeWidth 0 so the sliding line will not be visible")
    end
    line(leftX, y, rightX, y)
    
    -- Draw knob
    ellipse(sensor.parent.x, y, knobRadius*2)
    
    -- Text color
    local textCol = textColorOverride or color(stroke())
    
    fill(textCol)
    textMode(CENTER)
    
    -- Draw min, current, max above the slider
    local valStr = string.format("%.2f", _IMUI.sliderState[name])
    text(valStr, x, textY)
    
    textAlign(LEFT)
    text(tostring(min), leftX, textY)
    
    textAlign(RIGHT)
    text(tostring(max), rightX, textY)
    
    popStyle()
    
end

function showPopupContainer(cx, cy, w, h, drawFn)
    assert(type(drawFn) == "function", "showPopup requires a draw function")
    _IMUI.popupContainerLastDrawn = ElapsedTime
    
    -- Store popup box for dismissal check
    _IMUI.popupContainer = {
        x = cx - w/2,
        y = cy - h/2,
        w = w,
        h = h
    }
    
    -- Draw popup background (centered)
    pushStyle()
    rectMode(CENTER)
    roundedRectangle{
        x = cx,
        y = cy,
        w = w,
        h = h,
        corners = 15,
        radius = 18
    }
    popStyle()
    
    -- Draw contents with (0,0) as bottom-left corner of popup
    pushMatrix()
    translate(cx - w/2, cy - h/2)
    _IMUI.inContainer = true
    drawFn()
    _IMUI.inContainer = false
    popMatrix()
end

-- A helper for geometry
function pointInRectByCurrentMode(px, py, cx, cy, w, h)
    if false then -- set to true for debugging
        pushStyle()
        noFill()
        stroke(193, 236, 67)
        strokeWidth(8)
        rect(cx, cy, w, h)
        popStyle()
    end
    
    local mode = rectMode()
    local left, right, bottom, top
    if mode == CORNER then
        left   = cx
        bottom = cy
        right  = cx + w
        top    = cy + h
    elseif mode == CENTER then
        left   = cx - w/2
        bottom = cy - h/2
        right  = cx + w/2
        top    = cy + h/2
    elseif mode == RADIUS then
        left   = cx - w
        bottom = cy - h
        right  = cx + w
        top    = cy + h
    else
        -- default to center mode
        left   = cx - w/2
        bottom = cy - h/2
        right  = cx + w/2
        top    = cy + h/2
    end
    
    return (px >= left and px <= right and py >= bottom and py <= top)
end

_IMUI.checkboxState = _IMUI.checkboxState or {}

-- checkbox(name, x, y, size, initial, callback)
function checkbox(name, x, y, size, initial, callback)
    assert(name, "checkbox: name is required")
    size = size or 24
    cornerRadius = size * 0.3
    
    -- derive a unique key from the provided name
    local key = name
    
    -- initialize stored state if needed
    if _IMUI.checkboxState[key] == nil then
        _IMUI.checkboxState[key] = initial and true or false
    end
    local checked = _IMUI.checkboxState[key]
    
    -- draw the box via button(), passing our explicit key
    pushStyle()
    strokeWidth(2)
    button(
    "",      -- no label
    x, y,    -- center
    size,    -- w
    size,    -- h
    function()
        -- toggle and store
        checked = not checked
        _IMUI.checkboxState[key] = checked
        if callback then callback(checked) end
    end,
    { radius = cornerRadius, key = key }
    )
    popStyle()
    
    -- draw the check-mark if needed
    if checked then
        pushStyle()
        strokeWidth(size * 0.15)
        local s2 = size * 0.3
        line(x - s2,       y,          x - s2 * 0.2, y - s2)
        line(x - s2 * 0.2, y - s2,     x + s2,       y + s2)
        popStyle()
    end
    
    return checked
end


--make style settings into a single string 
function styleString(fillCol, strokeCol, sw, hiFill, hiStroke)
    local function comp(c) return table.concat({c.r|0,c.g|0,c.b|0,c.a|0},":") end
    local effSW = math.max(0, sw or 0)
    -- If effSW==0, treat stroke color as fill for the sig, same as we render
    local effStroke = (effSW == 0) and fillCol or strokeCol
    return comp(fillCol).."|"..comp(effStroke).."|"..string.format("%.3f", effSW)
    .."|"..comp(hiFill or effStroke).."|"..comp(hiStroke or fillCol)
end

function _buildButtonSpecs(w,h,r,corners, fillCol, strokeCol, sw, hiFill, hiStroke)
    local base = _getBaseRRect(w,h,r,corners)
    local sc   = base.shader and base.shader.scale or (0.25 / math.max(2, r))
    
    local function cloneWith(fillC, strokeC, strokeW)
        local c = mesh()
        c.vertices = base.vertices
        c.normals  = base.normals
        c.shader   = shader(rrectshad.vert, rrectshad.frag)
        c.shader.scale = sc
        
        -- 🔧 Mirror old behavior: if stroke is disabled, force stroke=fill and width=0
        if (strokeW or 0) <= 0 then
            strokeC = fillC
            strokeW = 0
        end
        
        c.shader.fillColor   = fillC
        c.shader.strokeColor = strokeC
        c.shader.strokeWidth = math.min(1 - sc*3, (strokeW or 0) * sc)
        return c
    end
    
    -- normal: use provided fill/stroke
    local normal = cloneWith(fillCol, strokeCol, sw)
    
    -- active/pressed:
    -- if hiFill/hiStroke provided, use those; else swap
    local pressedFill   = hiFill   or strokeCol
    local pressedStroke = hiStroke or fillCol
    local active = cloneWith(pressedFill, pressedStroke, sw)
    
    return {
        w=w, h=h, r=r, corners=corners,
        -- include active colors in the sig so we rebuild if they change
        styleString   = styleString(fillCol, strokeCol, sw, pressedFill, pressedStroke),
        normalMesh = normal,
        activeMesh = active
    }
end

-- fetch or rebuild a spec if geometry or style changed
function _getButtonSpecs(key, w,h,r,corners, fillCol, strokeCol, sw, hiFill, hiStroke)
    local p = _IMUI._btnSpecs[key]
    local need = (not p)
    or p.w ~= w or p.h ~= h or p.r ~= r or p.corners ~= corners
    or p.styleString ~= styleString(fillCol, strokeCol, sw, hiFill or strokeCol, hiStroke or fillCol)
    
    if need then
        p = _buildButtonSpecs(w,h,r,corners, fillCol, strokeCol, sw, hiFill, hiStroke)
        _IMUI._btnSpecs[key] = p
    end
    return p
end

function _rrectKey(w,h,r,corners, mode)
    -- this must match roundedRectangle()’s label construction exactly
    return table.concat({
        math.max(w+1, 2*r) + 1,
        math.max(h,  2*r) + 2,
        r, corners,
        CENTER, 0, 0
    }, ",")
end

function _getBaseRRect(w,h,r,corners)
    local key = _rrectKey(w,h,r,corners,CENTER)
    local base = __RRects[key]
    if not base then
        -- Prewarm cache without drawing
        local prevMode = rectMode()
        pushStyle()
        rectMode(CENTER)
        __RRects.__prewarm = true
        roundedRectangle{ x=0, y=0, w=w, h=h, radius=r, corners=corners }
        __RRects.__prewarm = nil
        popStyle()
        rectMode(prevMode)
        base = __RRects[key]
    end
    return base  -- this is the cached mesh with vertices/normals and shader.scale set
end





